package org.clazzes.svc.util.component;

import java.util.concurrent.ScheduledExecutorService;

import org.clazzes.jackson.rpc2.JsonRpcServlet;
import org.clazzes.svc.api.ComponentSupport;
import org.clazzes.svc.api.CoreService;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ServiceRegistry;
import org.clazzes.svc.util.voc.FieldAccessStrategy;
import org.clazzes.util.aop.ProxyFactory;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.ObjectMapper;

import jakarta.servlet.Servlet;

/** ServiceListener to be called e.g. once the respective datasource connection
 *  has been set up.
 *
 *  For Components with database access, implementors of this class set up
 *  the application logic, i.e. what was beans, services, etc. before.
 *
 *  As of writing, technically, this class can also be used for waiting for
 *  non-datasource listeners, however we might put data source specific
 *  code here if necessary in the future.
 *
 * @param <T> type we waited for
 */
public abstract class AbstractDataSourceListener<T> extends ComponentSupport implements ServiceListener<T> {
    protected ServiceContext serviceContext;
    protected ServiceRegistry serviceRegistry;

    public AbstractDataSourceListener(ServiceContext serviceContext) {
        this.serviceContext = serviceContext;
        this.serviceRegistry = this.serviceContext.getService(ServiceRegistry.class).get();
    }

    /**
     * Create a basic object mapper based on the given accessStrategy.
     * @param accessStrategy An access strategy.
     * @return The new object mapper suitable for further configuration.
     */
    public static ObjectMapper createObjectMapper(FieldAccessStrategy accessStrategy) {

        ObjectMapper objectMapper = new ObjectMapper();
        // Default in a newly created ObjectMapper is FieldAccessStrategy.GETTER.
        if (accessStrategy == FieldAccessStrategy.FIELD) {
            objectMapper.setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.NONE);
            objectMapper.setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.ANY);
        }

        return objectMapper;
    }

    /** Adds a new Jackson-based JsonRpcServlet as a service, based on some common
     *  configuratin options.
     *
     * @param serviceUrl the service url, something like /erp/svc/activityService
     * @param target the service to be added
     * @param serviceInterface the service interface
     * @param requestThreadLocalKey the request thread local key
     * @param accessStrategy the FieldAccess strategy.  Our old JsonRpcServlet generally
     *                  serialized using the fields itself, however the default Jackson
     *                  behaviour is to serialize based on the getters, which most proabably
     *                  breaks our old code.  So use FieldAccessStrategy.FIELD here to get the
     *                  old behaviour.
     */
    protected void addJsonRpcServlet(String serviceUrl, Object target, Class<?> serviceInterface, String requestThreadLocalKey,
                                     FieldAccessStrategy accessStrategy) {

        ObjectMapper objectMapper = createObjectMapper(accessStrategy);

        this.addJsonRpcServlet(serviceUrl,target,serviceInterface,requestThreadLocalKey,objectMapper);
    }

    /** Adds a new Jackson-based JsonRpcServlet as a service, based on some common
     *  configuratin options.
     *
     * @param serviceUrl the service url, something like /erp/svc/activityService
     * @param target the service to be added
     * @param serviceInterface the service interface
     * @param requestThreadLocalKey the request thread local key
     * @param objectMapper A completely set up object mapper. This istance is used
     *         throughout the lifeme of the serlvet.
     */
    protected void addJsonRpcServlet(String serviceUrl, Object target, Class<?> serviceInterface, String requestThreadLocalKey,
                                     ObjectMapper objectMapper) {


        JsonRpcServlet rpcServlet = new JsonRpcServlet(serviceUrl, serviceInterface, target, requestThreadLocalKey, objectMapper);
        this.addService(this.serviceRegistry, serviceUrl, Servlet.class, rpcServlet);
    }

    protected void addServlet(String serviceUrl, Servlet target, String requestThreadLocalKey) {
    	this.addService(this.serviceRegistry, serviceUrl, Servlet.class, target);
    }

    /** Adds a service using a JsonRpcServlet and the given ProxyFactory.
     *  See above addJsonRpcServlet for more documentation.
     */
    protected <U> void addProxyJsonRpcServlet(ProxyFactory proxyFactory, String serviceUrl, U service, Class<U> serviceInterface,
                                                     String requestThreadLocalKey, FieldAccessStrategy fieldAccessStrategy) {
        U serviceProxy = proxyFactory.getTypedProxy(service, serviceInterface);
        this.addJsonRpcServlet(serviceUrl, serviceProxy, serviceInterface, requestThreadLocalKey, fieldAccessStrategy);
    }

    /** Adds a service using a JsonRpcServlet and the given ProxyFactory.
     *  See above addJsonRpcServlet for more documentation.
     */
    protected <U> void addProxyJsonRpcServlet(ProxyFactory proxyFactory, String serviceUrl, U service, Class<U> serviceInterface,
                                                     String requestThreadLocalKey, ObjectMapper objectMapper) {
        U serviceProxy = proxyFactory.getTypedProxy(service, serviceInterface);
        this.addJsonRpcServlet(serviceUrl, serviceProxy, serviceInterface, requestThreadLocalKey,objectMapper);
    }


    protected CoreService getCoreService() {
    	return this.serviceContext.getService(CoreService.class).get();
    }

    protected ScheduledExecutorService getScheduledExecutorService() {
        CoreService coreService = this.getCoreService();
        return coreService.getScheduledExecutorService();
    }
}
