package org.clazzes.svc.util.component;

import org.clazzes.svc.api.ComponentSupport;
import org.clazzes.svc.api.ConfigWrapper;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ServiceRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Abstract baseclass for ConfigListeners, keeping track of
 *  what needs to be done once the config of a Component has been
 *  set up.
 *
 * @param <C> type of the Component-specific configuration (contains
 *     variables for all configuration fields of the Component)
 */
public abstract class AbstractConfigListener<C> extends ComponentSupport implements ConfigListener {

	private static final Logger log = LoggerFactory.getLogger(AbstractConfigListener.class);

	/** Fetches the Component-specific configuration from the general
	 *  ConfigWrapper instance.  In this method, possibly existing
	 *  default values should be applied.
	 *
	 *  The returned instance should be a plain java object without
	 *  special default value logics applied, just containing the
	 *  effective configuration.
	 *
	 * @param config the general ConfigWrapper instance
	 * @return the effective, application specific, configuration.
	 */
	protected abstract C fetchConfig(ConfigWrapper config);

	/** Performs the application-specific setup tasks that need to
	 *  be done, once the configuration of the Component at hand
	 *  has been loaded.
	 *
	 *  For Components without database connection or similar things,
	 *  here, the application logics needs to be set up, i.e. what
	 *  was modelled as beans, services, and similar, in the old osgi
	 *  world.
	 *
	 *  For Components with database connection or similar, here
	 *  the corresponding ServiceListener waiting for the database
	 *  connection (or similar) needs to be set up.
	 *
	 *  NOTE: In case some Component needs to listen to multiple
	 *        services at this point (like two instead of just one
	 *        mandatory database connections), we need to invent
	 *        some appropriate concept / setup at this point.
	 *
	 * @param registry the ServiceRegistry
	 * @param config the application specific config delivered by
	 *               fetchConfig above.
	 */
	protected abstract void setupBasedOnConfig(ServiceRegistry registry, C config);

	protected ServiceContext serviceContext;
	protected ServiceRegistry serviceRegistry;

	public AbstractConfigListener(ServiceContext serviceContext) {
		this.serviceContext = serviceContext;
		this.serviceRegistry = this.serviceContext.getService(ServiceRegistry.class).get();
	}

	public ServiceRegistry getServiceRegistry() {
		return this.serviceRegistry;
	}

	protected <T> void addListener(String key, Class<T> iface, ServiceListener<T> serviceListener) {
		this.addListener(this.serviceRegistry.listen(key, iface,
				(T service) -> {
					serviceListener.handleAdd(this, service);
				},
				(T service) -> {
					serviceListener.handleRemove(this, service);
				}));
	}

	@Override
	public void accept(ConfigWrapper configWrapper) {
		this.close();

		C config = this.fetchConfig(configWrapper);
		this.setupBasedOnConfig(this.serviceRegistry, config);
	}

	@Override
    public synchronized void closeBusinessLogics() {
		if (log.isDebugEnabled()) {
			log.debug("ConfigListener services for PID [" + this.getPid() + "] have been shut down.");
		}

		this.removeAllServices(this.serviceRegistry);
	}

	@Override
	public void close() {
		// close DB connection listener.
		this.closeAllListeners();
		// close all active services.
		this.closeBusinessLogics();
	}
}
