/***********************************************************
 *
 * GOGO SSH support of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/
package org.clazzes.svc.runner.sshd;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

import org.apache.sshd.common.cipher.BuiltinCiphers;
import org.apache.sshd.common.keyprovider.FileKeyPairProvider;
import org.apache.sshd.common.keyprovider.KeyPairProvider;
import org.apache.sshd.core.CoreModuleProperties;
import org.apache.sshd.server.SshServer;
import org.apache.sshd.server.config.keys.AuthorizedKeysAuthenticator;
import org.clazzes.svc.api.Component;
import org.clazzes.svc.api.ComponentManager;
import org.clazzes.svc.api.ConfigWrapper;
import org.clazzes.svc.api.ConfigurationEngine;
import org.clazzes.svc.api.CoreService;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ServicePriority;
import org.clazzes.svc.api.ServiceRegistry;
import org.clazzes.svc.api.cmd.CommandSet;
import org.clazzes.svc.runner.sshd.cmd.SvcCommands;
import org.jline.builtins.ssh.ShellCommand;
import org.jline.builtins.ssh.ShellFactoryImpl;
import org.jline.terminal.TerminalBuilder;
import org.jline.terminal.spi.TerminalProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Bundle activator, which starts the SSH server once the  CommandProcessor comes up.
 */
@ServicePriority(8)
public class SshdComponent implements Component {

    public static final String PID = "org.clazzes.svc.runner.sshd";

    private static final Logger log = LoggerFactory.getLogger(SshdComponent.class);

    private ExecutorService executorService;
    private SshServer sshServer;

    public SshdComponent() {
    }

    @Override
    public void start(ServiceContext svcCtxt) throws Exception {

        CoreService coreService = svcCtxt.getService(CoreService.class).get();

        ServiceRegistry registry = svcCtxt.getService(ServiceRegistry.class).get();

        ConfigurationEngine confEngine = svcCtxt.getService(ConfigurationEngine.class).get();

        ComponentManager componentManager = svcCtxt.getService(ComponentManager.class).get();

        Path etcDir = coreService.getEtcDir();

        Path sshdEtcDir;

        if (etcDir == null) {
            log.warn("Will not start svc-runner SSHD service in Unit Test mode.");
            sshdEtcDir = null;
        }
        else {
            sshdEtcDir = etcDir.resolve("sshd.d");
        }

        confEngine.listen(PID,(config) -> {
            this.stopSshServer();
            this.startSshServer(svcCtxt,sshdEtcDir,config,registry);
            componentManager.commit();
        });

    }

    @Override
    public void stop(ServiceContext svcCtxt) throws Exception {
        this.stopSshServer();
    }


    private synchronized void startSshServer(ServiceContext svcCtxt, Path etcPathObj, ConfigWrapper config, ServiceRegistry registry) {

        if (this.sshServer != null) {
            if (log.isDebugEnabled()) {
                log.debug("GoGo SSH Server already started.");
            }
            return;
        }

        SshServer sshd = SshServer.setUpDefaultServer();

        int port = config.getInt("port",8108);
        sshd.setPort(port);

        int timeout = config.getInt("timeout",600000);

        sshd.getProperties().put(CoreModuleProperties.IDLE_TIMEOUT.getName(), timeout);

        String host = config.getString("host","127.0.0.1").trim();

        if (!host.isEmpty() && !"*".equals(host)) {

            sshd.setHost(host);
        }

        // set up a cipher suite list with no cbc cipher.
        sshd.setCipherFactories(Arrays.asList(
                    BuiltinCiphers.cc20p1305_openssh,
                    BuiltinCiphers.aes256ctr,
                    BuiltinCiphers.aes192ctr,
                    BuiltinCiphers.aes128ctr,
                    BuiltinCiphers.aes256gcm,
                    BuiltinCiphers.aes128gcm));


        if (log.isDebugEnabled()) {
            log.debug("GoGo SSH Server's etcPath resolved to [{}].",etcPathObj);
        }

        sshd.setPublickeyAuthenticator(new AuthorizedKeysAuthenticator(etcPathObj.resolve("authorized_keys")));

        String algo = config.getString("hostKeyAlgorithm","EdDSA");

        Path keyFile;

        if ("RSA".equals(algo)) {

            keyFile = etcPathObj.resolve("ssh_host_rsa_key");
        }
        else if ("EdDSA".equals(algo)) {

            keyFile = etcPathObj.resolve("ssh_host_ed25519_key");
        }
        else if ("EC".equals(algo)) {

            keyFile = etcPathObj.resolve("ssh_host_ecdsa_key");
        }
        else {
            log.error("Error starting GoGo SSH Server on address ["+host+":"+port+
                    "], ssh key algorithm ["+algo+"] is not supported.");
            return;
        }

        if (log.isDebugEnabled()) {

            log.debug("keyFile = {}",keyFile);
            log.debug("Files.exists(keyFile) = {}",Files.exists(keyFile));
        }

        if (!Files.exists(keyFile)) {

            log.info("Key File [{}] does not exist, trying to generate the host key.",keyFile);

            if ("RSA".equals(algo)) {

                int keySize = config.getInt("hostKeySize",3072);

                try {
                    HostKeyGenerator.generateRSAHostKey(keyFile,keySize);
                }
                catch (Exception e) {
                    log.error("Error generating RSA host key",e);
                    return;
                }
            }
            else if ("EdDSA".equals(algo)) {

                int keySize = config.getInt("hostKeySize",256);

                try {
                    HostKeyGenerator.generateEdDSAHostKey(keyFile,keySize);
                }
                catch (Exception e) {
                    log.error("Error generating EdDSA host key",e);
                    return;
                }
            }
            else if ("EC".equals(algo)) {

                int keySize = config.getInt("hostKeySize",256);

                try {
                    HostKeyGenerator.generateECDSAHostKey(keyFile,keySize);
                }
                catch (Exception e) {
                    log.error("Error generating ECDSA host key",e);
                    return;
                }
            }
            else {
                log.error("Error starting GoGo SSH Server on address ["+host+":"+port+
                        "], ssh key for algorithm ["+algo+"] cannot be generated.");
                return;
            }
        }

        log.info("Loading {} key file [{}]",algo,keyFile);
        KeyPairProvider kpp = new FileKeyPairProvider(keyFile);

        sshd.setKeyPairProvider(kpp);

        ExecutorService es = null;

        try {

            es = Executors.newCachedThreadPool(r -> new Thread(r,"svc-runner-sshd"));

            var module = SshdComponent.class.getModule();

            var handler = new GogoSshHandler();
            handler.setExecutor(es);
            handler.setVersion(module.getDescriptor().version().get().toString());

            var dynamicResolver = new GogoCommandResolver();
            dynamicResolver.setGetCommandSets(() -> registry.getAll(CommandSet.class));

            var builtinResolver = new GogoCommandResolver();
            var builtinCommandSets = Map.<String, CommandSet>of("svc", new SvcCommands(svcCtxt));
            builtinResolver.setGetCommandSets(() -> builtinCommandSets);

            var externalResolver = new ExternalCommandResolver();
            externalResolver.setExecutor(es);

            handler.setResolver(
                new ListResolver(
                    List.of(
                        dynamicResolver,
                        builtinResolver,
                        externalResolver
                    )
                )
            );

            IllegalStateException iav = new IllegalStateException("jline3");
            List<TerminalProvider> providers = TerminalBuilder.builder().getProviders(null,iav);

            List<String> providerNames = providers.stream().map(p -> p.name()).collect(Collectors.toList());

            log.info("Available jline3 terminal providers are {}",providerNames);

            Throwable[] supp = iav.getSuppressed();
            if (supp != null && supp.length > 0) {

                StringBuilder summary = new StringBuilder();

                for (int i=0;i<supp.length;) {
                    Throwable sup = supp[i++];
                    summary.append("\n  jline3 err #").append(i).append(": ");
                    summary.append(sup.toString());
                }

                log.warn("[{}] jline3 terminal providers did not load:{}",supp.length,summary);


                if (log.isDebugEnabled()) {
                    log.debug("jline3 TerminalProvider load error details:",iav);
                }
            }

            var shellFactory = new ShellFactoryImpl(handler.interactiveHandler());

            sshd.setShellFactory(shellFactory);

            sshd.setCommandFactory((channel, command) -> new ShellCommand(handler.executeHandler(), command));

            if (log.isInfoEnabled()) {
                log.info("Starting GoGo SSH Server on address ["+host+":"+port+"]...");
            }

            sshd.start();
            this.sshServer = sshd;
            this.executorService = es;

            if (log.isInfoEnabled()) {
                log.info("Successfully started GoGo SSH Server on address ["+host+":"+port+"].");
            }

        } catch (Throwable e) {
            log.error("Error starting GoGo SSH Server on address ["+host+":"+port+"]",e);

            if (es != null) {
                es.shutdownNow();
            }
        }
    }

    private synchronized void stopSshServer() {

        if (this.sshServer != null) {

            SshServer sshd = this.sshServer;
            this.sshServer = null;

            try {
                if (log.isInfoEnabled()) {
                    log.info("Stopping GoGo SSH Server on address ["+sshd.getHost()+":"+sshd.getPort()+"]...");
                }

                sshd.stop();

                if (log.isInfoEnabled()) {
                    log.info("Successfully stopped GoGo SSH Server on address ["+sshd.getHost()+":"+sshd.getPort()+"].");
                }

            } catch (Exception e) {
                log.warn("Error stopping GoGo SSH Server on address ["+sshd.getHost()+":"+sshd.getPort()+"]",e);
            }
        }
        else {
            if (log.isDebugEnabled()) {
                log.debug("GoGo SSH Server already stopped.");
            }
        }

        if (this.executorService != null) {
            this.executorService.shutdownNow();
            this.executorService = null;
        }

    }
}
