package org.clazzes.svc.runner.sshd;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.function.Function;
import java.util.function.ToIntFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.clazzes.svc.runner.sshd.CommandParser.AndOr;
import org.clazzes.svc.runner.sshd.CommandParser.AndOrMode;
import org.clazzes.svc.runner.sshd.CommandParser.Command;
import org.clazzes.svc.runner.sshd.CommandParser.DoubleQuotedPart;
import org.clazzes.svc.runner.sshd.CommandParser.FileRedirectionKind;
import org.clazzes.svc.runner.sshd.CommandParser.ParsedCommand;
import org.clazzes.svc.runner.sshd.CommandParser.Pipeline;
import org.clazzes.svc.runner.sshd.CommandParser.RedirectionKind;
import org.clazzes.svc.runner.sshd.CommandParser.SyncMode;
import org.clazzes.svc.runner.sshd.CommandParser.Word;
import org.clazzes.svc.runner.sshd.CommandParser.WordPart;

// The execution engine for our posix-like shell language.
public class ShellExecutionEngine implements Closeable {
    @SuppressWarnings("unused")
    private static final org.slf4j.Logger log = org.slf4j.LoggerFactory.getLogger(ShellExecutionEngine.class);

    private final ExecutorService executor;
    private final CommandResolver commandResolver;
    private final Subshell mainSubshell;
    public ShellExecutionEngine(ExecutorService executor, CommandResolver commandResolver, CommandEnvironment env) {
        this.executor = executor;
        this.commandResolver = commandResolver;
        this.mainSubshell = new Subshell(env);
    }

    private static final Map.Entry<Function<Subshell, CliCommand>, String> exitEntry = Map.entry(subshell -> subshell::builtinExit, "Exit this shell");

    private static final Map<String, Map.Entry<Function<Subshell, CliCommand>, String>> builtins = Map.of(
        "help", Map.entry(subshell -> subshell::builtinHelp, "Print a list of available commands"),
        "exit", exitEntry,
        "quit", exitEntry,
        "logout", exitEntry,
        ":q!", exitEntry,
        ":q", exitEntry,
        ":wq", exitEntry
    );

    private static final Map.Entry<PipedInputStream, PipedOutputStream> pipe() {
        try {
            var input = new PipedInputStream();
            var output = new PipedOutputStream(input);
            return Map.entry(input, output);
        } catch (IOException e) {
            // Should be impossible.
            throw new RuntimeException(e);
        }
    }

    private final class UnwantedSideEffectException extends RuntimeException {
        public UnwantedSideEffectException() {}
    }

    private static final int waitOnSubshellFuture(Future<Integer> fut) {
        while (true) {
            try {
                return fut.get();
            } catch (InterruptedException e) {
                fut.cancel(true);
            } catch (ExecutionException e) {
                log.warn("Unexpected exception in subshell {}", e.getCause());

                return 1;
            }
        }
    }

    private int executeSubshell(CommandEnvironment env, ToIntFunction<Subshell> executionFunction) {
        try (var subshell = new Subshell(env)) {
            return executionFunction.applyAsInt(subshell);
        } catch (ShellExitException e) {
            return e.getStatus();
        } catch (Exception e) {
            log.error("Unexpected exception in subshell", e);
            return 1;
        }
    }

    private final class Subshell implements Closeable, CommandResolver {
        @Override
        public CliCommand resolveCommand(String commandName, CommandEnvironment env) {
            var builtin = builtins.get(commandName);
            if (builtin == null) {
                return null;
            }
            return builtin.getKey().apply(this);
        }

        @Override
        public List<CommandInfo> listCommands(String commandNamePrefix, CommandEnvironment env) {
            return builtins
                .entrySet()
                .stream()
                .filter(entry -> entry.getKey().startsWith(commandNamePrefix))
                .map(entry -> new CommandInfo(entry.getKey(), entry.getValue().getValue()))
                .toList();
        }

        public CommandEnvironment env;

        private List<Future<Integer>> jobs = new ArrayList<>();

        private CommandResolver subshellResolver = new ListResolver(List.of(commandResolver, this));

        public Subshell(CommandEnvironment env) {
            this.env = env;
        }

        private int builtinHelp(CommandParameters params) throws Exception {
            try (var writer = params.env().stdoutWriter()) {
                if (params.arguments().size() == 2) {
                    if (List.of("-h", "--help").contains(params.arguments().get(1))) {
                        writer.println("""
Usage: help [-h] [command]
Print a list of available commands or print the help of a specific command.

  -h, --help   display this help and exit""");
                        return 0;
                    } else {
                        var commandName = params.arguments().get(1);
                        var resolved = subshellResolver.resolveCommand(commandName, env);
                        if (resolved == null) {
                            writer.println(commandName + " not found");
                            return 1;
                        }

                        return resolved.execute(new CommandParameters(
                            params.env(),
                            List.of(
                                commandName,
                                "--help"
                            )
                        ));
                    }
                } else if (params.arguments().size() == 1) {
                    Iterable<CommandInfo> commands = () -> subshellResolver
                        .listCommands("", params.env())
                        .stream()
                        .sorted(Comparator.comparing(CommandInfo::name))
                        .iterator();
                    for (var commandInfo: commands) {
                        writer.println(commandInfo.name() + " - " + commandInfo.helpText());
                    }
                    return 0;
                } else {
                    writer.println("""
help: bad usage
Try 'help --help' for more information.
""");
                    return 1;
                }
            }
        }

        private int builtinExit(CommandParameters params) throws Exception {
            if (params.arguments().size() == 2 && List.of("-h", "--help").contains(params.arguments().get(1))) {
                try (var writer = params.env().stdoutWriter()) {
                    writer.println("""
Usage: %s [-h] [status]
Exit this shell with the given status, or 0 if none is given.

  -h, --help   display this help and exit"""
                        .formatted(params.arguments().get(0)));
                }
                return 0;
            }

            Integer status = null;
            if (params.arguments().size() == 1) {
                status = 0;
            } else if (params.arguments().size() == 2) {
                try {
                    status = Integer.valueOf(params.arguments().get(1));
                } catch (NumberFormatException e) {
                }
            }

            if (status == null) {
                try (var writer = params.env().stdoutWriter()) {
                    writer.println("""
%1$s: bad usage
Try '%1$s --help' for more information.
"""
                        .formatted(params.arguments().get(0)));
                }
                return 1;
            }

            throw new ShellExitException(status);
        }

        private String executeCommandSubstitution(ParsedCommand cmd) {
            var bytes = new ByteArrayOutputStream();
            executeSubshell(
                env.duplicate()
                    .withStdout(PosixFileDescriptor.ofOutput(bytes)),
                subshell -> subshell.executeCommand(cmd)
            );
            return new String(bytes.toByteArray(), StandardCharsets.UTF_8);
        }

        private String expandDoubleQuotedPart(DoubleQuotedPart part, boolean sideEffects) throws Exception {
            if (part instanceof DoubleQuotedPart.QuotedLiteral lit) {
                return lit.str();
            } else if (part instanceof DoubleQuotedPart.CommandSubstitution cmd) {
                if (!sideEffects) {
                    throw new UnwantedSideEffectException();
                }
                return executeCommandSubstitution(cmd.command());
            } else {
                throw new RuntimeException("Unsupported kind of double quoted part.");
            }
        }

        private String expandWordPart(WordPart part, boolean sideEffects) throws Exception {
            // TODO: ifs/in-field splitting
            if (part instanceof WordPart.Literal lit) {
                return lit.str();
            } else if (part instanceof WordPart.CommandSubstitution cmd) {
                if (!sideEffects) {
                    throw new UnwantedSideEffectException();
                }
                return executeCommandSubstitution(cmd.command());
            } else if (part instanceof WordPart.DoubleQuoted dq) {
                return dq.parts()
                    .stream()
                    .map(ExceptionFnUtil.sneakyFn(dqPart -> expandDoubleQuotedPart(dqPart, sideEffects)))
                    .collect(Collectors.joining(""));
            } else {
                throw new RuntimeException("Unsupported kind of word part.");
            }
        }

        public String expandWord(Word word, boolean sideEffects) {
            return word.parts()
                .stream()
                .map(ExceptionFnUtil.sneakyFn(part -> expandWordPart(part, sideEffects)))
                .collect(Collectors.joining(""));
        }

        public String expandWord(Word word) {
            return expandWord(word, true);
        }

        private PosixFileDescriptor handleFileRedirection(Path filePath, FileRedirectionKind kind) throws IOException {
            return switch (kind) {
                case APPEND -> PosixFileDescriptor.ofOutput(
                    Files.newOutputStream(
                        filePath,
                        StandardOpenOption.CREATE,
                        StandardOpenOption.APPEND,
                        StandardOpenOption.WRITE
                    )
                );
                case OUTPUT, OUTPUT_FORCE_CLOBBER -> PosixFileDescriptor.ofOutput(
                    Files.newOutputStream(filePath)
                );
                case INPUT -> PosixFileDescriptor.ofInput(
                    Files.newInputStream(filePath)
                );
            };
        }


        public PosixFileDescriptor handleRedirection(RedirectionKind kind) throws IOException {
            if (kind instanceof RedirectionKind.HereDocument hereDoc) {
                return PosixFileDescriptor.ofInput(new ByteArrayInputStream(hereDoc.document().getBytes(StandardCharsets.UTF_8)));
            } else if (kind instanceof RedirectionKind.Duplicate dup) {
                var prevFdNumber = Integer.valueOf(expandWord(dup.duplicatedFd()));
                var prev = this.env.fds().get(prevFdNumber);
                if (prev == null) {
                    throw new RuntimeException("bad file descriptor");
                }

                return (switch (dup.kind()) {
                    case INPUT -> prev.onlyInput();
                    case OUTPUT -> prev.onlyOutput();
                    case BOTH -> prev;
                }).duplicate();
            } else if (kind instanceof RedirectionKind.File file) {
                var filePath = Paths.get(this.env.cwd()).resolve(expandWord(file.word()));

                return handleFileRedirection(filePath, file.kind());
            } else {
                throw new RuntimeException("unsupported kind of redirection");
            }
        }

        public int executeSimple(Command.Simple simple) {
            var mainWordExpanded = expandWord(simple.mainWord());

            var argv = Stream.concat(
                Stream.of(mainWordExpanded),
                simple.otherWords()
                    .stream()
                    .map(ExceptionFnUtil.sneakyFn(this::expandWord))
            ).toList();

            var newEnvVars = new HashMap<>(this.env.envVars());
            for (var assignment: simple.assignments()) {
                newEnvVars.put(assignment.key(), expandWord(assignment.value()));
            }

            var newFds = new HashMap<>(this.env.duplicate().fds());

            try {
                for (var redirection: simple.redirections()) {
                    var newFd = handleRedirection(redirection.kind());
                    var prevFd = this.env.fds().get(redirection.fd());
                    if (prevFd != null) {
                        prevFd.close();
                    }
                    newFds.put(redirection.fd(), newFd);
                }
            } catch (Exception e) {
                for (var newFd: newFds.values()) {
                    newFd.close();
                }

                try {
                    env.stderr().write(("osgi-shell: " + e.getMessage() + "\n").getBytes(StandardCharsets.UTF_8));
                } catch (IOException e1) {
                    log.warn("Unable to write error message", e1);
                }
                return 1;
            }

            try (var newEnv = new CommandEnvironment(newEnvVars, newFds, this.env.cwd())) {
                var resolved = subshellResolver.resolveCommand(mainWordExpanded, newEnv);
                if (resolved == null) {
                    try {
                        newEnv.stderr().write(("osgi-shell: command not found: " + mainWordExpanded + "\n").getBytes(StandardCharsets.UTF_8));
                    } catch (IOException e1) {
                        log.warn("Unable to write error message", e1);
                    }
                    return 127;
                } else {
                    try {
                        return resolved.execute(new CommandParameters(newEnv, argv));
                    } catch (ShellExitException e) {
                        throw e;
                    } catch (Exception e) {
                        newEnv.printException(e);
                        return 1;
                    }
                }
            }
        }

        public int executeCommand(Command command) {
            if (command instanceof Command.Simple simple) {
                return this.executeSimple(simple);
            } else {
                throw new RuntimeException("Unsupported kind of command.");
            }
        }

        public int executePipeline(Pipeline pipeline) {
            assert pipeline.commands().size() > 0;

            if (pipeline.commands().size() == 1) {
                return this.executeCommand(pipeline.commands().get(0));
            }

            List<Future<Integer>> futures = new ArrayList<>();
            var currentPipeInput = env.fds().get(0);
            var firstCommands = pipeline.commands().subList(0, pipeline.commands().size() - 1);
            for (var command: firstCommands) {
                var pipe = pipe();

                var fork = env
                    .duplicate()
                    .withStdin(currentPipeInput)
                    .withStdout(PosixFileDescriptor.ofOutput(pipe.getValue()));

                futures.add(executor.submit(
                    () -> executeSubshell(fork, subshell -> subshell.executeCommand(command))
                ));

                currentPipeInput = PosixFileDescriptor.ofInput(pipe.getKey());
            }

            var lastFork =
                env
                    .duplicate()
                    .withStdin(currentPipeInput)
            ;
            var lastCommand = pipeline.commands().get(pipeline.commands().size() - 1);
            var lastFuture = executor.submit(
                () -> executeSubshell(lastFork, subshell -> subshell.executeCommand(lastCommand))
            );

            int ret = waitOnSubshellFuture(lastFuture);

            for (var future: futures) {
                waitOnSubshellFuture(future);
            }

            return ret;
        }

        public int executeAndOr(AndOr andOr) {
            int lastStatus = this.executePipeline(andOr.firstPipeline());
            for (var element: andOr.elements()) {
                if (element.mode() == AndOrMode.AND) {
                    if (lastStatus == 0) {
                        lastStatus = this.executePipeline(element.pipeline());
                    }
                } else {
                    if (lastStatus != 0) {
                        lastStatus = this.executePipeline(element.pipeline());
                    }
                }
            }

            return lastStatus;
        }

        public int executeCommand(ParsedCommand command) {
            int lastStatus = 0;
            for (var element: command.list()) {
                if (element.mode() == SyncMode.ASYNC) {
                    var fork = this.env.duplicate();
                    this.jobs.add(executor.submit(() -> executeSubshell(fork, subshell -> subshell.executeAndOr(element.andOr()))));

                    lastStatus = 0;
                } else {
                    lastStatus = this.executeAndOr(element.andOr());
                }
            }
            return lastStatus;
        }

        @Override
        public void close() {
            this.env.close();
        }
    }

    public int executeCommand(ParsedCommand command) {
        return this.mainSubshell.executeCommand(command);
    }

    public CommandEnvironment getEnv() {
        return this.mainSubshell.env;
    }

    public CommandResolver getResolver() {
        return this.mainSubshell.subshellResolver;
    }

    // Try to expand a word without side-effects.
    // This is useful for completions and highlighting.
    // There are no guarantees about what kind of shell constructs this functions is able to expand.
    public Optional<String> expandWordWithoutSideEffects(Word word) {
        try {
            return Optional.of(this.mainSubshell.expandWord(word, false));
        } catch (UnwantedSideEffectException e) {
            return Optional.empty();
        }
    }

    @Override
    public void close() throws IOException {
        this.mainSubshell.close();
    }
}
