package org.clazzes.svc.runner.sshd;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import org.jline.utils.AttributedString;
import org.jline.utils.AttributedStringBuilder;
import org.jline.utils.AttributedStyle;
import org.clazzes.parsercombinators.ParseException;
import org.jline.reader.EndOfFileException;
import org.jline.reader.LineReaderBuilder;
import org.jline.reader.impl.completer.ArgumentCompleter;
import org.jline.terminal.Terminal;

public class InteractiveShell {
    private static final AttributedString bold(String str) {
        return new AttributedString(str, AttributedStyle.BOLD);
    }

    private static final AttributedString mkBranding(String version) {
        // TODO: Investigate line endings and crlf on windows.
        return new AttributedStringBuilder()
            .style(AttributedStyle.DEFAULT.foreground(AttributedStyle.CYAN))
            // Replacing backslashes with box drawing characters for escaped readability.
            .append("""
     _________  __ ____           _______ __ __  ____   ____   ___________
    /  ___/╲  ╲/ // ___╲   ______ ╲_  __ ╲  |  ╲/    ╲ /    ╲_/ __ ╲_  __ ╲
    ╲___ ╲  ╲   /╲  ╲___  /_____/  |  | ╲/  |  /   |  ╲   |  ╲  ___/|  | ╲/
   /_____/   ╲_/  ╲_____|          |__|  |____/|___|  /___|  /╲___  >__|
                                                    ╲/     ╲/     ╲/
""".replace('╲', '\\').stripTrailing())
            .style(AttributedStyle.DEFAULT)
            .append("\n\n")
            .append(bold("  clazzes.org svc-runner SSH console")).append(" (").append(version).append(")\n")
            .append("\n")
            .append("Hit '").append(bold("help")).append("' for a list of available commands\n")
            .append("and '").append(bold("help [cmd]")).append("' for help on a specific command.\n")
            .append("Hit '").append(bold("<ctrl-d>")).append("' or type '").append(bold("logout")).append("' or '").append(bold("exit")).append("' to disconnect shell from current session.\n")
            .toAttributedString();
    }

    public static final void runInteractiveShell(
        String version,
        ExecutorService executor,
        CommandResolver resolver,
        Terminal terminal,
        Map<String, String> envVars,
        String cwd
    ) {

        var terminalFd = PosixFileDescriptor.ofTerminal(terminal);

        try (var shell = new ShellExecutionEngine(
            executor,
            resolver,
            new CommandEnvironment(
                envVars,
                Map.of(
                    0, terminalFd.duplicate(),
                    1, terminalFd.duplicate(),
                    2, terminalFd.duplicate()
                ),
                cwd
            )
        )) {

            mkBranding(version).print(terminal);

            var completer = new CommandCompleter(shell);

            var lineReader = LineReaderBuilder.builder()
                .terminal(terminal)
                .highlighter(new CommandHighlighter(shell))
                .expander(new NoopExpander())
                .parser(completer)
                .completer(completer)
                .completionMatcher(new CommandCompletionMatcher())
                .build();

            var user = envVars.get("USER");

            while (true) {
                try {
                    var command = lineReader.readLine(user + ":" + shell.getEnv().cwd() + "$ ");
                    try {
                        shell.executeCommand(CommandParser.parseCommand(command));
                    } catch (ShellExitException e) {
                        break;
                    } catch (ParseException e) {
                        var redColoredErrorMessage = new AttributedString(
                            e.getError().getCompositeMessage(),
                            AttributedStyle.DEFAULT.foreground(AttributedStyle.RED)
                        );

                        redColoredErrorMessage.println(terminal);
                    }
                } catch (EndOfFileException e) {
                    break;
                }
            }

        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
