/***********************************************************
 *
 * GOGO SSH support of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

 package org.clazzes.svc.runner.sshd;

import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.DosFileAttributeView;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermissions;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.spec.ECGenParameterSpec;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.apache.sshd.common.config.keys.writer.openssh.OpenSSHKeyPairResourceWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.i2p.crypto.eddsa.spec.EdDSAGenParameterSpec;

public abstract class HostKeyGenerator {

    private static final Logger log = LoggerFactory.getLogger(HostKeyGenerator.class);

    protected static String makeComment() {

        DateFormat df = new SimpleDateFormat("yyyyMMdd");

        return "gogo-ssh-" + df.format(new Date());
    }

    protected static void writePivateKey(KeyPair kp, String comment, Path file) throws Exception {

        OpenSSHKeyPairResourceWriter osshkprw = new OpenSSHKeyPairResourceWriter();

        osshkprw.writePrivateKey(kp,comment, null,
                                 Files.newOutputStream(file,
                                                       StandardOpenOption.CREATE_NEW,
                                                       StandardOpenOption.WRITE));

        PosixFileAttributeView posix = Files.getFileAttributeView(file,PosixFileAttributeView.class);

        if (posix != null) {
            posix.setPermissions(PosixFilePermissions.fromString("r--r-----"));
        }
        else {
            DosFileAttributeView dos = Files.getFileAttributeView(file,DosFileAttributeView.class);

            if (dos != null) {
                dos.setReadOnly(true);
            }
        }
     }

    public static String generateECDSAHostKey(Path file, int keySize) throws Exception {

        String comment = makeComment();

        String algorithm;

        switch (keySize) {

            case 256:
            algorithm = "secp256r1";
            break;

            case 384:
            algorithm = "secp384r1";
            break;

            case 521:
            algorithm = "secp521r1";
            break;

            default:
            throw new IllegalArgumentException("EC key size ["+keySize+"] is not in the set {256, 384, 521}.");
        }

        log.info("Generating [{}] ECDSA host key with comment [{}]...",algorithm,comment);

        KeyPairGenerator kpg = KeyPairGenerator.getInstance("EC");

        kpg.initialize(new ECGenParameterSpec(algorithm));

        KeyPair kp = kpg.generateKeyPair();

        log.info("Writing ECDSA host key with comment [{}] to [{}].",
                comment,file);

        writePivateKey(kp,comment,file);

        return comment;
    }

    public static String generateEdDSAHostKey(Path file, int keySize) throws Exception {

        String comment = makeComment();

        String algorithm;

        switch (keySize) {

            case 256:
            algorithm = "Ed25519";
            break;
/*
    Not yet supported by eddsa-0.3.0
            case 456:
            algorithm = "X448";
            break;
 */
            default:
            throw new IllegalArgumentException("EdEC key size ["+keySize+"] is not 256.");
        }

        log.info("Generating [{}] ECDSA host key with comment [{}]...",algorithm,comment);

        // this works only, if eddsa-0.3.0 is deployed
        KeyPairGenerator kpg = KeyPairGenerator.getInstance("EdDSA","EdDSA");

        kpg.initialize(new EdDSAGenParameterSpec(algorithm));

        KeyPair kp = kpg.generateKeyPair();

        log.info("Writing EdDSA host key with comment [{}] to [{}].",
                comment,file);

        writePivateKey(kp,comment,file);

        return comment;
    }

    public static String generateRSAHostKey(Path file, int keySize) throws Exception {

        if (keySize < 2048 || keySize > 8192) {
            throw new IllegalArgumentException("RSA key size ["+keySize+"] is not in the range [2048, 8192].");
        }

        String comment = makeComment();

        log.info("Generating RSA-{} host key with comment [{}]...",keySize,comment);

        KeyPairGenerator kpg = KeyPairGenerator.getInstance("RSA");

        kpg.initialize(keySize);

        KeyPair kp = kpg.generateKeyPair();

        log.info("Writing RSA host key with comment [{}] to [{}].",
                comment,file);

        writePivateKey(kp,comment,file);

        return comment;
    }
}
