package org.clazzes.svc.runner.sshd;

import java.io.Closeable;
import java.io.File;
import java.io.FileDescriptor;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.ProcessBuilder.Redirect;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.stream.Stream;

import org.jline.terminal.Terminal;
import org.jline.terminal.impl.AbstractPosixTerminal;
import org.jline.terminal.impl.AbstractPty;
import org.jline.terminal.impl.jansi.JansiNativePty;
import org.jline.terminal.impl.jna.JnaNativePty;
import org.jline.terminal.impl.jni.JniNativePty;

// This resolver allows us to access standard linux cli tools.
public class ExternalCommandResolver implements CommandResolver {
    @SuppressWarnings("unused")
    private static final org.slf4j.Logger log = org.slf4j.LoggerFactory.getLogger(ExternalCommandResolver.class);

    private ExecutorService executor;
    public void setExecutor(ExecutorService executor) {
        this.executor = executor;
    }

    // We need to extract the raw stdin stream from the jline terminal because the jline terminal inputstream swallows ctrl-d EOFs.
    private static final FileDescriptor getRawStdin(Terminal terminal) {
        if (!(terminal instanceof AbstractPosixTerminal posixTerm) || posixTerm.getPty() == null) {
            log.warn("Unable to get raw stdin from terminal, because terminal is of unknown type {}.", terminal.getClass().getName());
            return null;
        }

        var pty = posixTerm.getPty();

        if (pty instanceof AbstractPty abstractPty && abstractPty.getSystemStream() != null) {
            return switch (abstractPty.getSystemStream()) {
                case Input -> FileDescriptor.in;
                case Output -> FileDescriptor.out;
                case Error -> FileDescriptor.err;
            };
        } else if (pty instanceof JniNativePty jniPty) {
            return jniPty.getSlaveFD();
        } else if (pty instanceof JnaNativePty jnaPty) {
            return jnaPty.getSlaveFD();
        } else if (pty instanceof JansiNativePty jansiPty) {
            return jansiPty.getSlaveFD();
        } else if (pty.getClass().getName().equals("org.jline.terminal.impl.ffm.FfmNativePty")) {
            Method getSlaveFdMethod;
            try {
                getSlaveFdMethod = pty.getClass().getMethod("getSlaveFD");
            } catch (NoSuchMethodException | SecurityException e) {
                log.warn("Couldn't get getSlaveFD function from FfmNativePty", e);
                return null;
            }
            FileDescriptor slaveFd;
            try {
                slaveFd = (FileDescriptor) getSlaveFdMethod.invoke(pty);
            } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
                log.warn("Error while calling getSlaveFD function", e);
                return null;
            }
            return slaveFd;
        } else {
            log.warn("Unable to get raw stdin from terminal, because the pty has unknown type {}.", pty.getClass().getName());
            return null;
        }
    }

    private void doTransfer(InputStream in, OutputStream out, Closeable closeable) {
        SelectionKey key = null;
        SelectableChannel channel = null;
        try {
        try (
            var _c = closeable;
            // If we get a raw FileInputStream, we improve the interrupt-ability of this thread by polling the filedescriptor before reading.
            // The symptom that this fixes is the terminal swallowing a single typed character directly after executing an external command.
            var selector = in instanceof FileInputStream ? Selector.open() : null;
        ) {
            if (in instanceof FileInputStream file) {
                channel = jdk.nio.Channels.readWriteSelectableChannel(
                    file.getFD(),
                    new jdk.nio.Channels.SelectableChannelCloser() {
                        @Override
                        public void implCloseChannel(SelectableChannel arg0) throws IOException {
                            // NOTHING
                        }

                        @Override
                        public void implReleaseChannel(SelectableChannel arg0) throws IOException {
                            // NOTHING
                        }
                    }
                );
                channel.configureBlocking(false);
                key = channel.register(selector, SelectionKey.OP_READ);
            }

            var buffer = new byte[8192];
            while (true) {
                if (key != null) {
                    selector.select();
                    if (Thread.interrupted()) {
                        // We where interrupted.
                        break;
                    }

                    if (!key.isReadable()) {
                        continue;
                    }
                }

                int read = in.read(buffer);
                if (read < 0) {
                    break;
                }
                out.write(buffer, 0, read);
                out.flush();
            }
        } finally {
            if (channel != null) {
                channel.configureBlocking(true);
            }
        }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private CliCommand command(String commandPath) {
        return params -> {
            // TODO: Calling external programs properly requires calling libc functions directly.
            var processBuilder = new ProcessBuilder(
                Stream.concat(
                    Stream.of(commandPath),
                    params
                        .arguments()
                        .stream()
                        .skip(1)
                ).toList()
            )
                .redirectError(Redirect.PIPE)
                .redirectInput(Redirect.PIPE)
                .redirectOutput(Redirect.PIPE)
                .directory(new File(params.cwd()));
            processBuilder.environment().clear();
            processBuilder.environment().putAll(params.envVars());

            var stdinFd = params.fds().get(0);
            Objects.requireNonNull(stdinFd, "No stdin connected.");
            Objects.requireNonNull(stdinFd.in(), "Stdin isn't readable");
            var process = processBuilder.start();

            var stdin = Optional.ofNullable(stdinFd.terminal())
                .map(term -> getRawStdin(term))
                .<InputStream>map(fd -> new FileInputStream(fd))
                .orElse(stdinFd.in());

            // If we cancel a future we can't actually wait for it's completion using get() anymore.
            // But we need to wait for the future to make the stdin file descriptor blocking before we give it back to the linereader again.
            var latch = new CountDownLatch(1);

            var future1 = this.executor.submit(() -> {
                try {
                    doTransfer(
                        stdin,
                        process.getOutputStream(),
                        process.getOutputStream()
                    );
                } finally {
                    latch.countDown();
                }
            });
            var future2 = this.executor.submit(() -> doTransfer(
                process.getInputStream(),
                params.stdout(),
                process.getInputStream()
            ));
            var future3 = this.executor.submit(() -> doTransfer(
                process.getErrorStream(),
                params.stderr(),
                process.getErrorStream()
            ));

            var status = process.waitFor();
            future1.cancel(true);
            future2.get();
            future3.get();
            try {
                future1.get();
            } catch (CancellationException e) {
                log.debug("Cancelled", e);
            }
            latch.await();
            process.getInputStream().close();
            process.getErrorStream().close();
            process.getOutputStream().close();

            return status;
        };
    }

    private String[] getPath(CommandEnvironment env) {
        var env_path = System.getenv("PATH");
        return env_path.split(":");
    }

    @Override
    public CliCommand resolveCommand(String commandName, CommandEnvironment env) {
        if (commandName.startsWith("/")) {
            if (Files.isExecutable(Path.of(commandName))) {
                return command(commandName);
            } else {
                return null;
            }
        }

        // TODO: Should we use the CommandEnvironment?
        for (var path: getPath(env)) {
            var combinedPath = Path.of(path).resolve(commandName);
            if (Files.isExecutable(combinedPath)) {
                return command(combinedPath.toString());
            }
        }

        return null;
    }

    private Stream<Path> listDirectoryIfExists(Path dir) throws IOException {
        if (!Files.isDirectory(dir)) {
            return Stream.empty();
        } else {
            return Files.list(dir);
        }
    }

    @Override
    public List<CommandInfo> listCommands(String commandNamePrefix, CommandEnvironment env) {
        try {
            // We aren't going to do help or completion for every command in $PATH
            if (commandNamePrefix == null || commandNamePrefix.isEmpty()) {
                return List.of();
            }

            var isPath = commandNamePrefix.startsWith("/");
            if (isPath) {
                var lastSlashIndex = commandNamePrefix.lastIndexOf('/');
                var directory = commandNamePrefix.substring(0, lastSlashIndex);
                var searchPrefix = commandNamePrefix.substring(lastSlashIndex);

                return listDirectoryIfExists(Path.of(directory))
                    .filter(path -> path.getFileName().toString().startsWith(searchPrefix))
                    .map(completion -> directory + completion.getFileName())
                    .map(name -> new CommandInfo(name, "External command"))
                    .toList();
            } else {
                return Arrays.stream(getPath(env))
                    .flatMap(ExceptionFnUtil.sneakyFn(path -> listDirectoryIfExists(Path.of(path))))
                    .filter(path -> path.getFileName().toString().startsWith(commandNamePrefix))
                    .map(path -> path.getFileName().toString())
                    .distinct()
                    .map(commandName -> new CommandInfo(commandName, "External command"))
                    .toList();
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
