package org.clazzes.svc.runner.sshd;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

public record CommandEnvironment(
    Map<String, String> envVars,
    Map<Integer, PosixFileDescriptor> fds,
    String cwd
) implements Closeable {
    public CommandEnvironment(CommandEnvironment other) {
        this(other.envVars(), other.fds(), other.cwd());
    }

    public InputStream stdin() {
        return Objects.requireNonNull(Objects.requireNonNull(fds.get(0), "No stdin stream defined").in());
    }

    public OutputStream stdout() {
        return Objects.requireNonNull(Objects.requireNonNull(fds.get(1), "No stdout stream defined").out());
    }

    public OutputStream stderr() {
        return Objects.requireNonNull(Objects.requireNonNull(fds.get(2), "No stderr stream defined").out());
    }

    public CommandEnvironment withFds(Map<Integer, PosixFileDescriptor> fds) {
        return new CommandEnvironment(envVars, fds, cwd);
    }

    // This closes the previous occupant.
    public CommandEnvironment withFd(int key, PosixFileDescriptor value) {
        if (fds().get(key) != null) {
            fds().get(key).close();
        }

        var newFds = new HashMap<>(fds());
        newFds.put(key, value);
        return this.withFds(newFds);
    }

    // This closes the previous occupant.
    public CommandEnvironment withStdout(PosixFileDescriptor value) {
        return this.withFd(1, value);
    }

    // This closes the previous occupant.
    public CommandEnvironment withStdin(PosixFileDescriptor value) {
        return this.withFd(0, value);
    }

    // Copy and increase the refcount of our fds.
    public CommandEnvironment duplicate() {
        return new CommandEnvironment(
            envVars,
            fds
                .entrySet()
                .stream()
                .map(entry -> Map.entry(entry.getKey(), entry.getValue().duplicate()))
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)),
            cwd
        );
    }

    @Override
    public void close() {
        for (var posixFd: fds.values()) {
            posixFd.close();
        }
    }

    public PrintWriter fdWriter(int fd) {
        return new PrintWriter(new CloseProtectedOutputStream(this.fds.get(fd).out()), false, StandardCharsets.UTF_8);
    }

    public PrintWriter stderrWriter() {
        return this.fdWriter(2);
    }

    public PrintWriter stdoutWriter() {
        return this.fdWriter(1);
    }

    public void printException(Throwable e) {
        try (var writer = stdoutWriter()) {
            e.printStackTrace(writer);
        }
    }


}
