package org.clazzes.svc.runner.monitoring.system;

import java.nio.file.FileStore;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.clazzes.svc.api.monitoring.HealthInfo;
import org.clazzes.svc.api.monitoring.HealthInfoWithLog;
import org.clazzes.svc.api.monitoring.HealthStatus;
import org.clazzes.svc.api.util.SvcUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DiskHealthCheck extends SystemHealthCheckSupport {

    private static final Logger log = LoggerFactory.getLogger(DiskHealthCheck.class);

    public DiskHealthCheck(double warningPercentage, double criticalPercentage) {
        super(warningPercentage,criticalPercentage);
    }

    @Override
    public HealthInfo call() throws Exception {

        FileSystem fileSystem = FileSystems.getDefault();

        List<HealthInfo> infos = new ArrayList<HealthInfo>();

        for (Path path : fileSystem.getRootDirectories()) {

            FileStore store = Files.getFileStore(path);

            if (log.isDebugEnabled()) {
                log.debug("store={},total={},usable={}",
                        store,store.getTotalSpace(),store.getUsableSpace());
            }

            long totalGB = store.getTotalSpace();
            long freeGB = store.getUsableSpace();

            double percentage = 100.0 * (1.0 - (double)freeGB/(double)totalGB);

            if (percentage < this.warningPercentage) {
                infos.add(new HealthInfo(HealthStatus.OK,
                    String.format(Locale.ENGLISH,
                        "Path [%s] usage [%.1f%%<%.1f%%], free [%s/%s]",
                        path,
                        percentage,this.warningPercentage,
                        SvcUtil.bytesHumanReadable(freeGB),
                        SvcUtil.bytesHumanReadable(totalGB))));
            }
            else if (percentage < this.criticalPercentage) {

                infos.add(new HealthInfo(HealthStatus.WARN,
                    String.format(Locale.ENGLISH,
                        "Path [%s] usage [%.1f%%\u2264%.1f%%<%.1f%%], free [%s/%s]",
                        path,
                        this.warningPercentage,percentage,this.criticalPercentage,
                        SvcUtil.bytesHumanReadable(freeGB),
                        SvcUtil.bytesHumanReadable(totalGB))));
            }
            else {

                infos.add(new HealthInfo(HealthStatus.CRITICAL,
                    String.format(Locale.ENGLISH,
                        "Path [%s] usage [%.1f%%\u2264%.1f%%], free [%s/%s]",
                        path,
                        this.criticalPercentage,percentage,
                        SvcUtil.bytesHumanReadable(freeGB),
                        SvcUtil.bytesHumanReadable(totalGB))));
            }
        }

        if (infos.size() == 0) {
            return new HealthInfo(HealthStatus.HEALTH_CHECK_ERROR,
                            "Cannot find any root directory.");
        }
        else {
            return HealthInfoWithLog.ofLog("Disk usage",infos);
        }

    }

    @Override
    public String getId() {
        return "system.disk";
    }

    @Override
    public String getDescription() {
        return "Check the Disk usage against Thresholds [warn="+
                this.warningPercentage+"%,critical="+
                this.criticalPercentage+"%]";
    }

    @Override
    public List<String> getTags() {
        return List.of("system","disk");
    }

}
