/***********************************************************
 *
 * Service Runner framework runner using commons-daemon
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.runner.monitoring.openmetrics;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.Map.Entry;
import java.util.SortedMap;
import java.util.TreeMap;

import org.clazzes.svc.runner.monitoring.IActiveMetrics;
import org.clazzes.svc.runner.monitoring.IActiveMonitoring;

import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServlet;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;

public class OpenMetricsServlet extends HttpServlet {

    // see
    // https://github.com/prometheus/OpenMetrics/blob/main/specification/OpenMetrics.md#overall-structure
    private static final String CONTENT_TYPE="application/openmetrics-text;version=1.0.0;charset=utf-8";
    private static final String UI_CONTENT_TYPE="text/plain;charset=utf-8";

    private final IActiveMonitoring monitoring;

    public OpenMetricsServlet(IActiveMonitoring monitoring) {
        this.monitoring = monitoring;
    }

    private static final void setNoCacheHeaders(HttpServletResponse resp) {
        resp.setHeader("Cache-Control","no-cache");
        resp.setHeader("Pragma","no-cache");
        resp.setHeader("Expires","0");
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {

        String[] ids = req.getParameterValues("id");
        String[] tags = req.getParameterValues("tag");
        boolean ui = req.getParameter("ui") != null;

        SortedMap<String,IActiveMetrics<?>> selectedMetrics = new TreeMap<String,IActiveMetrics<?>>();

        if (ids == null && tags == null) {
            selectedMetrics.putAll(this.monitoring.getHealthChecks());
            selectedMetrics.putAll(this.monitoring.getCounters());
            selectedMetrics.putAll(this.monitoring.getGauges());
        }
        else {
            if (ids != null) {
                for (String id:ids) {
                    this.monitoring.getById(id).ifPresent(m -> selectedMetrics.put(id,m));
                }
            }

            if (tags != null) {
                for (String tag : tags) {
                    for (IActiveMetrics<?> m : this.monitoring.getByTag(tag)) {
                        selectedMetrics.put(m.getTaggedMetrics().getId(),m);
                    }
                }
            }
        }


        resp.setContentType(ui ? UI_CONTENT_TYPE : CONTENT_TYPE);
        setNoCacheHeaders(resp);

        try (OutputStreamWriter w = new OutputStreamWriter(resp.getOutputStream())) {

            for (Entry<String, IActiveMetrics<?>> e : selectedMetrics.entrySet()) {
                OpenMetricsFormatter.formatMetrics(w,e.getValue());
            }
        }
    }

}
