/***********************************************************
 *
 * Service Runner framework runner using commons-daemon
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.runner.monitoring;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledExecutorService;

import org.clazzes.svc.api.monitoring.Counter;
import org.clazzes.svc.api.monitoring.Gauge;
import org.clazzes.svc.api.monitoring.HealthCheck;
import org.clazzes.svc.api.monitoring.HealthInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MonitoringEngine implements IActiveMonitoring {

    private static final Logger log = LoggerFactory.getLogger(MonitoringEngine.class);

    private final ScheduledExecutorService executorService;
    private final int keepResultsSeconds;

    private final ConcurrentHashMap<String,MetricsHolder<HealthInfo>> healthChecks;
    private final ConcurrentHashMap<String,MetricsHolder<Long>> counters;
    private final ConcurrentHashMap<String,MetricsHolder<Double>> gauges;

    private final ConcurrentHashMap<String,List<IActiveMetrics<?>>> tags;

    public MonitoringEngine(ScheduledExecutorService executorService, int keepResultsSeconds) {
        this.executorService = executorService;
        this.keepResultsSeconds = keepResultsSeconds;

        this.healthChecks = new ConcurrentHashMap<String,MetricsHolder<HealthInfo>>();
        this.counters = new ConcurrentHashMap<String,MetricsHolder<Long>>();
        this.gauges = new ConcurrentHashMap<String,MetricsHolder<Double>>();

        this.tags = new ConcurrentHashMap<String,List<IActiveMetrics<?>>>();
    }


    public void addToTags(IActiveMetrics<?> am) {

        List<String> myTags = am.getTaggedMetrics().getTags();

        if (myTags != null) {
            for (String tag : myTags) {
                this.tags.putIfAbsent(tag,new CopyOnWriteArrayList<IActiveMetrics<?>>());
                List<IActiveMetrics<?>> tagMetrics = this.tags.get(tag);
                tagMetrics.add(am);
            }
        }
    }

    public void removeFromTags(IActiveMetrics<?> am) {

        List<String> myTags = am.getTaggedMetrics().getTags();

        if (myTags != null) {
            for (String tag : myTags) {
                List<IActiveMetrics<?>> tagMetrics =
                    this.tags.putIfAbsent(tag,new CopyOnWriteArrayList<IActiveMetrics<?>>());
                tagMetrics.add(am);
            }
        }
    }
    public void addHealthCheck(String key, HealthCheck healthCheck) {

        log.info("Registering HealthCheck [{}] for key [{}].",
                     healthCheck.getClass().getName(),key);

        MetricsHolder<HealthInfo> holder =
            MetricsHolder.ofHealthCheck(healthCheck,keepResultsSeconds);

        if (holder.schedule(this.executorService)) {
            log.info("Successfully started async HealthCheck [{}] for key [{}].",
                     healthCheck.getClass().getName(),key);
        }

        MetricsHolder<HealthInfo> old = this.healthChecks.put(key,holder);

        if (old != null) {
            log.warn("Overwriting old HealthCheck [{}] for key [{}].",
                     old.getClass().getName(),key);
            old.shutdown();
        }

        this.addToTags(holder);
    }

    public void removeHealthCheck(String key) {

        MetricsHolder<HealthInfo> holder = this.healthChecks.remove(key);

        if (holder != null) {
            log.warn("Removing HealthCheck [{}] for key [{}].",
                     holder.getTaggedMetrics().getClass().getName(),key);
            holder.shutdown();
            this.removeFromTags(holder);
        }
        else {
            log.warn("No HealthCheck to remove for key [{}].",key);
        }
    }

    @Override
    public Map<String,IActiveMetrics<HealthInfo>> getHealthChecks() {
        return new HashMap<String,IActiveMetrics<HealthInfo>>(this.healthChecks);
    }

    @Override
    public IActiveMetrics<HealthInfo> getHealthCheck(String key) {
        return this.healthChecks.get(key);
    }

    public void addCounter(String key, Counter counter) {

        log.info("Registering Counter [{}] for key [{}].",
                     counter.getClass().getName(),key);

        MetricsHolder<Long> holder =
            MetricsHolder.ofCounter(counter,keepResultsSeconds);

        if (holder.schedule(this.executorService)) {
            log.info("Successfully started async Counter [{}] for key [{}].",
                     counter.getClass().getName(),key);
        }

        MetricsHolder<Long> old = this.counters.put(key,holder);

        if (old != null) {
            log.warn("Overwriting old Counter [{}] for key [{}].",
                     old.getClass().getName(),key);
            old.shutdown();
        }

        this.addToTags(holder);
    }

    public void removeCounter(String key) {

        MetricsHolder<Long> holder = this.counters.remove(key);

        if (holder != null) {
            log.warn("Removing Counter [{}] for key [{}].",
                     holder.getTaggedMetrics().getClass().getName(),key);
            holder.shutdown();
            this.removeFromTags(holder);
        }
        else {
            log.warn("No Counter to remove for key [{}].",key);
        }
    }

    @Override
    public Map<String,IActiveMetrics<Long>> getCounters() {
        return new HashMap<String,IActiveMetrics<Long>>(this.counters);
    }

    @Override
    public IActiveMetrics<Long> getCounter(String key) {
        return this.counters.get(key);
    }

    public void addGauge(String key, Gauge gauge) {

        log.info("Registering Gauge [{}] for key [{}].",
                     gauge.getClass().getName(),key);

        MetricsHolder<Double> holder =
            MetricsHolder.ofGauge(gauge,keepResultsSeconds);

        if (holder.schedule(this.executorService)) {
            log.info("Successfully started async Gauge [{}] for key [{}].",
                     gauge.getClass().getName(),key);
        }

        MetricsHolder<Double> old = this.gauges.put(key,holder);

        if (old != null) {
            log.warn("Overwriting old Gauge [{}] for key [{}].",
                     old.getClass().getName(),key);
            old.shutdown();
        }

        this.addToTags(holder);
    }

    public void removeGauge(String key) {

        MetricsHolder<Double> holder = this.gauges.remove(key);

        if (holder != null) {
            log.warn("Removing Gauge [{}] for key [{}].",
                     holder.getTaggedMetrics().getClass().getName(),key);
            holder.shutdown();
            this.removeFromTags(holder);
        }
        else {
            log.warn("No Gauge to remove for key [{}].",key);
        }
    }

    @Override
    public Map<String,IActiveMetrics<Double>> getGauges() {
        return new HashMap<String,IActiveMetrics<Double>>(this.gauges);
    }

    @Override
    public IActiveMetrics<Double> getGauge(String key) {
        return this.gauges.get(key);
    }

    @Override
    public List<IActiveMetrics<?>> getByTag(String tag) {
        List<IActiveMetrics<?>> ret = this.tags.get(tag);

        if (ret == null) {
            ret = Collections.emptyList();
        }

        return ret;
    }


    @Override
    public Optional<IActiveMetrics<?>> getById(String id) {

        IActiveMetrics<?> ret = this.healthChecks.get(id);
        if (ret != null) {
            return Optional.of(ret);
        }

        ret = this.counters.get(id);
        if (ret != null) {
            return Optional.of(ret);
        }

        ret = this.gauges.get(id);
        if (ret != null) {
            return Optional.of(ret);
        }

        return Optional.empty();
    }


}
