/***********************************************************
 *
 * GOGO JDBC commands of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.runner.jdbc.cmd;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.PrintStream;
import java.io.Reader;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.NClob;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.AbstractMap;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.sql.DataSource;

import org.clazzes.svc.api.CoreService;
import org.clazzes.svc.api.ServiceRegistry;
import org.clazzes.svc.api.cmd.Argument;
import org.clazzes.svc.api.cmd.CommandSet;
import org.clazzes.svc.api.cmd.Descriptor;
import org.clazzes.svc.api.cmd.Parameter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * JDBC commands for the GoGo console.
 */
public class JdbcCommands implements CommandSet {

    private static final Logger log = LoggerFactory.getLogger(JdbcCommands.class);

    public static final List<String> COMMANDS = List.of("connect","connection","desc","dump","lsc","query","showtables","sql");

    private static final String[] DEFAULT_SHOWTABLE_RESULT_COLUMNS = new String[] {
            "TABLE_NAME","TABLE_TYPE","TABLE_CAT","TABLE_SCHEM"
    };
    private static final String[] DEFAULT_DESC_RESULT_COLUMNS = new String[] {
            "COLUMN_NAME","TYPE_NAME","COLUMN_SIZE","DECIMAL_DIGITS","NULLABLE"
    };

    private static final char[] HEX_DIGITS= new char[] {'0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F'};

    private static final String NULL_STRING = "\u00A0";

    private static final Pattern SECRET_PATTERN = Pattern.compile("^secret::(env|prop|void):(.*)$");

    private final CoreService coreService;
    private final ServiceRegistry serviceRegistry;

    public JdbcCommands(CoreService coreService, ServiceRegistry serviceRegistry) {

        this.coreService = coreService;
        this.serviceRegistry = serviceRegistry;
    }

    @Override
    public List<String> getCommands() {

        return COMMANDS;
    }

    @Descriptor("List registered JDBC connections.")
    public void lsc(PrintStream out) throws Exception {

        Map<String,DataSource> datasources = this.serviceRegistry.getAll(DataSource.class);

        out.println("Registered Datasources:");

        if (datasources == null || datasources.isEmpty()) {
            out.println("  No datasources found.");
        }
        else {

            for (String name : datasources.keySet()) {

                if (name != null) {
                    out.println("  "+name);
                }
            }
        }
    }

    protected static final void printDatabaseMetaData(PrintStream out,
                                    String datasourceName,
                                    Connection conn) throws SQLException {

        DatabaseMetaData md = conn.getMetaData();

        out.println("Datasource ["+datasourceName+"] properties:");
        out.println("  JDBC URL:        " + md.getURL());
        out.println("  Product Name:    " + md.getDatabaseProductName());
        out.println("  Product Version: " + md.getDatabaseProductVersion());
        out.println("  Driver Name:     " + md.getDriverName());
        out.println("  Driver Version:  " + md.getDriverVersion());
        out.println("  QuoteString:     " + md.getIdentifierQuoteString());
    }

    @Descriptor("Show connection informations.")
    public void connection(PrintStream out,
            @Descriptor("datasource name") String datasourceName) throws Exception {

        try (DataSourceContext ctx = new DataSourceContext(this.serviceRegistry, datasourceName);
             Connection conn = ctx.getDataSource().getConnection()) {

            printDatabaseMetaData(out,datasourceName,conn);
        }
    }

    @Descriptor("Connect to a database by URL, username and password.")
    public void connect(PrintStream out,
            @Descriptor("JDBC URL") String url,
            @Descriptor("Connection properties like user=myuser or password=secret::env:MY_PASS") String[] props) throws Exception {

        Properties properties = new Properties(props.length);

        Map<String,Entry<String,String>> secrets = new HashMap<String,Entry<String,String>>(props.length);

        for (String prop:props) {

            String[] kv = prop.strip().split("\s*=\s*",2);

            String k = kv[0];
            String v;

            if (kv.length < 2) {
                v = null;
                out.println("WARNING: key ["+k+"] given without value, assuming null.");
            }
            else {
                v = kv[1];
                Matcher m = SECRET_PATTERN.matcher(v);

                if (m.matches()) {
                    String scheme = m.group(1);
                    String key = m.group(2);

                    secrets.put(k,
                        new AbstractMap.SimpleImmutableEntry<String,String>(scheme,key));

                    v = this.coreService.getSecret("org.clazzes.jdbc.provider",scheme,key);
                }
            }

            properties.put(k,v);
        }

        boolean oracle = url.startsWith("jdbc:oracle:");

        out.println("Connecting to database ["+url+"]...");

        try (Connection conn = DriverManager.getConnection(url,properties)) {

            printDatabaseMetaData(out,url,conn);

            out.println("YAML configuration will be:");
            out.println("database:");
            out.println("  DBNAME:");

            out.println("      url: " + maybeQuoteYaml(url));

            out.println("      validationQuery: " +
                (oracle ? "select 1 from dual" : "select 1")
            );

            for (Object k : properties.keySet()) {

                Entry<String,String> secret = secrets.get(k);

                String yamlKey = "user".equals(k) ? "username" : k.toString();

                if (secret == null) {
                    Object v = properties.get(k);
                    out.println("      "+yamlKey+": " + maybeQuoteYaml(v.toString()));
                }
                else {
                    out.println("      "+yamlKey+":");
                    out.println("        service.type: secret");
                    out.println("        scheme: "+ secret.getKey());
                    out.println("        key: "+ maybeQuoteYaml(secret.getValue()));
                }
            }
        }
    }

    private static final void appendHexByte(StringBuilder sb, byte b) {
        sb.append(HEX_DIGITS[(b>>4)&0xf]);
        sb.append(HEX_DIGITS[b&0xf]);
    }

    private static final String formatValue(ResultSetMetaData md, ResultSet rs, int idx, int maxWidth) throws SQLException {

        String o = null;
        int type = md.getColumnType(idx);

        switch(type) {

        case Types.DECIMAL:
        case Types.NUMERIC:
        case Types.BIT:
        case Types.BOOLEAN:
        case Types.TINYINT:
        case Types.SMALLINT:
        case Types.INTEGER:
        case Types.FLOAT:
        case Types.DOUBLE:
        case Types.REAL:
        case Types.BIGINT:
        case Types.VARCHAR:
        case Types.NVARCHAR:
        case Types.CHAR:
        case Types.NCHAR:
        case Types.TIME:
        case Types.TIMESTAMP:
        case Types.DATE:
        case Types.LONGVARCHAR:
        case Types.LONGNVARCHAR:
            o = rs.getString(idx);
            break;

        case Types.BINARY:
        case Types.VARBINARY:

            byte[] b = rs.getBytes(idx);

            if (b!=null) {

                if (b.length == 0) {
                    o = "";
                }
                else {
                    int nbytes = (maxWidth-2)/2;

                    StringBuilder sb = new StringBuilder(maxWidth);
                    sb.append("0x");

                    for (int i=0;i<b.length && i<nbytes;++i) {
                        appendHexByte(sb,b[i]);
                    }

                    o=sb.toString();
                }
            }
            break;

        // BLOBs, CLOBs, TEXTs,...
        case Types.BLOB:
        case Types.LONGVARBINARY:

            Blob blob = rs.getBlob(idx);

            if (blob != null) {

                int nbytes = (maxWidth-2)/2;

                if (blob.length() < nbytes) {
                    nbytes = (int)blob.length();
                }

                if (nbytes <= 0) {

                    o = "";
                }
                else {
                    StringBuilder sb = new StringBuilder(maxWidth);
                    sb.append("0x");

                    b = blob.getBytes(0L,nbytes);

                    for (int i=0;i<b.length && i<nbytes;++i) {
                        appendHexByte(sb,b[i]);
                    }

                    o=sb.toString();
                }
            }
            break;

        case Types.CLOB:

            Clob clob = rs.getClob(idx);

            if (clob != null) {

                int nchars = maxWidth;

                if (clob.length() < nchars) {
                    nchars = (int)clob.length();
                }

                o = clob.getSubString(0L,nchars);
            }
            break;

        case Types.NCLOB:

            NClob nclob = rs.getNClob(idx);

            if (nclob != null) {

                int nchars = maxWidth;

                if (nclob.length() < nchars) {
                    nchars = (int)nclob.length();
                }

                o = nclob.getSubString(0L,nchars);
            }
            break;

        default:
            o = String.format(Locale.ENGLISH,"<unknown type %d>",type);
            break;
        }

        return o;
    }

    private static final void printRawValue(PrintStream out, Charset cs, ResultSetMetaData md, ResultSet rs, int i) throws SQLException, IOException {

        int type = md.getColumnType(i+1);

        switch(type) {

        case Types.DECIMAL:
        case Types.NUMERIC:
        case Types.BIT:
        case Types.BOOLEAN:
        case Types.TINYINT:
        case Types.SMALLINT:
        case Types.INTEGER:
        case Types.FLOAT:
        case Types.DOUBLE:
        case Types.REAL:
        case Types.BIGINT:
        case Types.VARCHAR:
        case Types.NVARCHAR:
        case Types.CHAR:
        case Types.NCHAR:
        case Types.TIME:
        case Types.TIMESTAMP:
        case Types.DATE:
        case Types.LONGVARCHAR:
        case Types.LONGNVARCHAR:

            String s = rs.getString(i+1);
            if (s == null) {
                out.write('\000');
            }
            else {
                out.write(rs.getString(i+1).getBytes(cs));
            }
            break;

        // BLOBs, CLOBs, TEXTs,...
        case Types.LONGVARBINARY:
        case Types.VARBINARY:
        case Types.BINARY:

            byte[] b = rs.getBytes(i+1);

            if (b == null) {
                out.write('\000');
            }
            else {
                out.write(b);
            }
            break;

        case Types.BLOB:

            Blob blob = rs.getBlob(i+1);

            if (blob == null) {
                out.write('\000');
            }
            else {

                try (InputStream is = blob.getBinaryStream()) {

                    byte[] buf = new byte[16384];
                    int n;

                    while ((n=is.read(buf)) > 0) {
                        out.write(buf,0,n);
                    }
                }
            }
            break;

        case Types.CLOB:
        case Types.NCLOB:

            Clob clob = rs.getClob(i+1);

            if (clob == null) {
                out.write('\000');
            }
            else {

                try (Reader reader = clob.getCharacterStream()) {

                    CharsetEncoder enc = cs.newEncoder();
                    CharBuffer buf = CharBuffer.allocate(8192);
                    ByteBuffer bbuf = ByteBuffer.allocate(16384);

                    int n;

                    while ((n=reader.read(buf.array(),buf.position(),buf.remaining())) > 0) {

                        buf.limit(buf.position()+n);
                        buf.position(0);

                        enc.encode(buf,bbuf,false);

                        out.write(bbuf.array(),0,bbuf.position());
                        bbuf.position(0);

                        buf.compact();
                    }

                    if (buf.position() > 0) {

                        buf.limit(buf.position());
                        buf.position(0);

                        enc.encode(buf,bbuf,true);

                        out.write(bbuf.array(),0,bbuf.position());
                        bbuf.position(0);
                    }
                }
            }
            break;

        default:
            out.write(String.format(Locale.ENGLISH,"<unknown type %d>",type).getBytes(cs));
            break;
        }
    }

    private static final String quoteCsv(String v) {

        StringBuilder sb = new StringBuilder(v.length() * 2);

        sb.append('"');

        for (int i=0;i<v.length();i  = v.offsetByCodePoints(i,1)) {

            int cp = v.codePointAt(i);

            if (cp == '"') {
                // perform quoting
                sb.append("\"\"");
            }
            else {
                sb.appendCodePoint(cp);
            }
        }

        sb.append('"');

        return sb.toString();
    }

    private static final String maybeQuoteCsv(String v) {

        if (v == null) {
            return "";
        }

        if (v.isEmpty()) {
            return "\"\"";
        }

        for (int i=0;i<v.length();i  = v.offsetByCodePoints(i,1)) {

            int cp = v.codePointAt(i);

            if (cp == '"' || cp == ',' || cp == '\n' || cp == '\r') {
                // perform quoting
                return quoteCsv(v);
            }
        }

        return v;
    }

    private static final String quoteYaml(String v) {

        StringBuilder sb = new StringBuilder(v.length() * 2);

        sb.append('"');

        for (int i=0;i<v.length();i  = v.offsetByCodePoints(i,1)) {

            int cp = v.codePointAt(i);

            if (cp == '"' || cp == '\\') {
                // perform quoting
                sb.append('\\');
            }
            sb.appendCodePoint(cp);
        }

        sb.append('"');

        return sb.toString();
    }

    private static final Set<Integer> YAML_SPECIALS =
        ":{}[],&*#?|-<>=!%@\\".chars().boxed().collect(Collectors.toSet());

    private static final String maybeQuoteYaml(String v) {

        if (v == null) {
            return "";
        }

        if (v.isEmpty()) {
            return "\"\"";
        }

        for (int i=0;i<v.length();i  = v.offsetByCodePoints(i,1)) {

            int cp = v.codePointAt(i);

            if (YAML_SPECIALS.contains(cp)) {
                // perform quoting
                return quoteYaml(v);
            }
        }

        return v;
    }

    private static void printResultSetCsv(PrintStream out, ResultSet rs, int maxWidth) throws SQLException {

        ResultSetMetaData md = rs.getMetaData();

        int n = md.getColumnCount();

        StringBuilder line = new StringBuilder(16384);

        for (int i=0;i<n;++i) {

            if (i>0) {
                line.append(',');
            }
            line.append(maybeQuoteCsv(md.getColumnName(i+1)));
        }
        out.println(line.toString());

        while (rs.next()) {

            line.setLength(0);
            for (int i=0;i<n;++i) {

                if (i>0) {
                    line.append(',');
                }
                String o = formatValue(md,rs,i+1,maxWidth);
                line.append(maybeQuoteCsv(o));
            }
            out.println(line.toString());
        }
    }

    private static void printResultSetRaw(PrintStream out, ResultSet rs, String encoding) throws SQLException, IOException {

        Charset cs = Charset.forName(encoding);

        ResultSetMetaData md = rs.getMetaData();

        int n = md.getColumnCount();
        int irow = 0;

        while (rs.next()) {

            if (irow > 0) {
                // ASCII record separator.
                out.write('\u001e');
            }

            for (int i=0;i<n;++i) {

                if (i>0) {
                    // ASCII unit separator.
                    out.write('\u001f');
                }
                printRawValue(out,cs,md,rs,i);
            }
            ++irow;
        }
    }

    private static final String pad(String o, int width) {

        String v = o == null ? "null" : o;

        if (v.length() == width) {
            return v;
        }

        if (v.length() > width) {
            return v.substring(0,width-1) + "\u2026";
        }

        StringBuilder sb = new StringBuilder(width);

        sb.append(v);

        while (sb.length() < width) {
            sb.append(' ');
        }

        return sb.toString();
    }

    private static void printResultSet(PrintStream out, ResultSet rs, int maxWidth, String[] columns) throws SQLException {

        ResultSetMetaData md = rs.getMetaData();

        int n = md.getColumnCount();

        int widths[] = new int[n];
        int indices[];

        if (columns == null) {

            indices = new int[n];

            for (int i=0;i<n;++i) {
                indices[i] = i+1;
            }
        }
        else {

            indices = new int[columns.length];

            Map<String,Integer> nameColumns = new HashMap<String,Integer>();

            for (int i=0;i<n;++i) {

                nameColumns.put(md.getColumnName(i+1),i+1);
            }

            n=0;

            for (int i=0;i<columns.length;++i) {

                Integer idx = nameColumns.get(columns[i]);

                if (idx == null) {
                    log.warn("Column [{}] not found in result column list [{}]",columns[i],nameColumns.keySet());
                }
                else {
                    indices[n++] = idx.intValue();
                }
            }
        }

        int nall = 1;

        for (int i=0;i<n;++i) {

            int type = md.getColumnType(indices[i]);

            switch(type) {

            case Types.VARCHAR:
            case Types.NVARCHAR:
            case Types.CHAR:
            case Types.NCHAR:
            case Types.TIME:
            case Types.TIMESTAMP:
            case Types.DATE:
                widths[i] = md.getPrecision(indices[i]);
                break;

            case Types.DECIMAL:
            case Types.NUMERIC:
                widths[i] = md.getPrecision(indices[i])+1;
                break;

            case Types.BIT:
            case Types.BOOLEAN:
            case Types.TINYINT:
            case Types.SMALLINT:
                widths[i] = 6;
                break;
            case Types.INTEGER:
                widths[i] = 10;
                break;
            case Types.FLOAT:
            case Types.DOUBLE:
            case Types.BIGINT:
                widths[i] = 20;
                break;

            // BLOBs, CLOBs, TEXTs,...
            default:
                widths[i] = maxWidth;
                break;
            }

            int l = md.getColumnName(indices[i]).length();

            if (l > widths[i]) {
                widths[i] = l;
            }

            if (widths[i] > maxWidth) {
                widths[i] = maxWidth;
            }

            nall += widths[i];
            ++nall;

            if (log.isDebugEnabled()) {
                log.debug("Column {}: type={}, width={}",new Object[]{indices[i],type,widths[i]});
            }
        }

        StringBuilder line = new StringBuilder(nall);

        line.setLength(0);
        line.append('|');
        for (int i=0;i<n;++i) {

            line.append(pad(md.getColumnName(indices[i]),widths[i]));
            line.append('|');
        }
        out.println(line.toString());

        line.setLength(0);
        line.append('+');
        for (int i=0;i<n;++i) {

            for (int j=0;j<widths[i];++j) {
                line.append('-');
            }
            line.append('+');
        }
        out.println(line.toString());

        int nlines = 0;

        while (rs.next()) {

            line.setLength(0);
            line.append('|');
            for (int i=0;i<n;++i) {

                // fetch two more values in order to get ellipsis for truncated blobl values.
                String o = formatValue(md,rs,indices[i],maxWidth+2);
                line.append(pad(o,widths[i]));
                line.append('|');
            }
            out.println(line.toString());


            ++nlines;
        }

        switch(nlines) {

        case 0:
            out.println("No results");
            break;

        case 1:
            out.println("1 result");
            break;

        default:
            out.println(String.format(Locale.ENGLISH,"%d results",nlines));
        }
    }

    @Descriptor("Show all database tables.")
    public void showtables(PrintStream out,
            @Descriptor("Database catalog to search") @Parameter(names={"-C", "--catalog"}, absentValue = NULL_STRING) String catalog,
            @Descriptor("Database Schema pattern") @Parameter(names={"-s", "--schema"}, absentValue = NULL_STRING) String schemaPattern,
            @Descriptor("Table Name pattern") @Parameter(names={"-t", "--table"}, absentValue = NULL_STRING) String tableNamePattern,
            @Descriptor("Maximal output length of column data") @Parameter(names={"-l", "--length"}, absentValue = "32") int outputLength,
            @Descriptor("Verbose output in normal output (print all JDBC columns)") @Parameter(names={"-v", "--verbose"}, presentValue="true", absentValue = "false") boolean verbose,
            @Descriptor("Output in CSV format") @Parameter(names={"-c", "--csv"}, presentValue="true", absentValue = "false") boolean csv,
            @Descriptor("Output in raw binary format") @Parameter(names={"-r", "--raw"}, presentValue="true", absentValue = "false") boolean raw,
            @Descriptor("Encoding of raw output") @Parameter(names={"-e", "--encoding"}, absentValue = "UTF-8") String encoding,
            @Descriptor("Name of the registered datasource") @Argument("datasource") String datasourceName
            ) throws Exception {

        if (NULL_STRING.equals(catalog)) {
            catalog = null;
        }
        if (NULL_STRING.equals(schemaPattern)) {
            schemaPattern = null;
        }
        if (NULL_STRING.equals(tableNamePattern)) {
            tableNamePattern = null;
        }

        try (DataSourceContext ctx = new DataSourceContext(this.serviceRegistry, datasourceName);
             Connection conn = ctx.getDataSource().getConnection()) {

            DatabaseMetaData md = conn.getMetaData();

            try (ResultSet rs = md.getTables(catalog, schemaPattern, tableNamePattern,null)) {

                if (csv) {
                    printResultSetCsv(out,rs,outputLength);
                }
                else if (raw) {
                    printResultSetRaw(out,rs,encoding);
                }
                else {
                    printResultSet(out,rs,outputLength, verbose ? null : DEFAULT_SHOWTABLE_RESULT_COLUMNS);
                }
            }
        }
    }

    @Descriptor("Describe a database table.")
    public void desc(PrintStream out,
            @Descriptor("Database catalog to search") @Parameter(names={"-C", "--catalog"}, absentValue = NULL_STRING) String catalog,
            @Descriptor("Database Schema pattern") @Parameter(names={"-s", "--schema"}, absentValue = NULL_STRING) String schemaPattern,
            @Descriptor("Maximal output length of column data") @Parameter(names={"-l", "--length"}, absentValue = "32") int outputLength,
            @Descriptor("Verbose output in normal output (print all JDBC columns)") @Parameter(names={"-v", "--verbose"}, presentValue="true", absentValue = "false") boolean verbose,
            @Descriptor("Output in CSV format") @Parameter(names={"-c", "--csv"}, presentValue="true", absentValue = "false") boolean csv,
            @Descriptor("Output in raw binary format") @Parameter(names={"-r", "--raw"}, presentValue="true", absentValue = "false") boolean raw,
            @Descriptor("Encoding of raw output") @Parameter(names={"-e", "--encoding"}, absentValue = "UTF-8") String encoding,
            @Descriptor("Name of the registered datasource") @Argument("datasource")  String datasourceName,
            @Descriptor("Table Name to descibe") @Argument("table") String tableName
            ) throws Exception {

        if (NULL_STRING.equals(catalog)) {
            catalog = null;
        }
        if (NULL_STRING.equals(schemaPattern)) {
            schemaPattern = null;
        }

        try (DataSourceContext ctx = new DataSourceContext(this.serviceRegistry, datasourceName);
             Connection conn = ctx.getDataSource().getConnection()) {

            DatabaseMetaData md = conn.getMetaData();

            try (ResultSet rs = md.getColumns(catalog,schemaPattern,tableName,null)) {

                if (csv) {
                    printResultSetCsv(out,rs,outputLength);
                }
                else if (raw) {
                    printResultSetRaw(out,rs,encoding);
                }
                else {
                    printResultSet(out,rs,outputLength, verbose ? null: DEFAULT_DESC_RESULT_COLUMNS);
                }
            }
        }
    }


    private static void runQuery(PrintStream out, Statement statement, String sql, int outputLength, boolean csv,boolean raw, String encoding) throws SQLException,IOException {

        if (log.isDebugEnabled()) {
            log.debug("Executing query [{}].",sql);
        }

        if (statement.execute(sql)) {

            int irs = 0;

            do {

                try (ResultSet rs = statement.getResultSet()) {

                    if (csv) {
                        printResultSetCsv(out,rs,outputLength);
                    }
                    else if (raw) {

                        if (irs > 0) {
                            // ASCII group separator.
                            out.write('\u001d');
                        }
                        printResultSetRaw(out,rs,encoding);
                    }
                    else {
                        printResultSet(out,rs,outputLength,null);
                    }
                }

                ++irs;
            }
            while (statement.getMoreResults());
        }
        else {
            int n = statement.getUpdateCount();

            if (n==0) {
                out.println("No row updated.");
            }
            else if (n==1) {
                out.println("1 row updated.");
            }
            else if (n>1) {
                out.println(String.format(Locale.ENGLISH,"%d rows updated.",n));
            }
        }
    }

    @Descriptor("Perform an SQL query.")
    public void query(PrintStream out,
            @Descriptor("Maximal output length of column data") @Parameter(names={"-l", "--length"}, absentValue = "32") int outputLength,
            @Descriptor("Maximal number of rows to show") @Parameter(names={"-L", "--limit"}, absentValue = "-1") int limit,
            @Descriptor("Output in CSV format") @Parameter(names={"-c", "--csv"}, presentValue="true", absentValue = "false") boolean csv,
            @Descriptor("Output in raw binary format") @Parameter(names={"-r", "--raw"}, presentValue="true", absentValue = "false") boolean raw,
            @Descriptor("Encoding of raw output") @Parameter(names={"-e", "--encoding"}, absentValue = "UTF-8") String encoding,
            @Descriptor("Name of the registered datasource") @Argument("datasource")  String datasourceName,
            @Descriptor("Query parts") @Argument("query") String query[]
            ) throws Exception {

        if (raw && csv) {
            throw new IllegalArgumentException("You must not simulatenously specify -r and -c, choose either CSV or raw output.");
        }

        try (DataSourceContext ctx = new DataSourceContext(this.serviceRegistry, datasourceName);
             Connection conn = ctx.getDataSource().getConnection();
             Statement statement = conn.createStatement()) {

            if (limit >= 0) {
                statement.setMaxRows(limit);
            }

            String sql = String.join(" ",query);
            runQuery(out,statement,sql,outputLength,csv,raw,encoding);
        }
    }

    @Descriptor("Dump a single database table.")
    public void dump(PrintStream out,
            @Descriptor("Maximal output length of column data") @Parameter(names={"-l", "--length"}, absentValue = "32") int outputLength,
            @Descriptor("Maximal number of rows to show") @Parameter(names={"-L", "--limit"}, absentValue = "-1") int limit,
            @Descriptor("Output in CSV format") @Parameter(names={"-c", "--csv"}, presentValue="true", absentValue = "false") boolean csv,
            @Descriptor("Output in raw binary format") @Parameter(names={"-r", "--raw"}, presentValue="true", absentValue = "false") boolean raw,
            @Descriptor("Encoding of raw output") @Parameter(names={"-e", "--encoding"}, absentValue = "UTF-8") String encoding,
            @Descriptor("Name of the registered datasource") @Argument("datasource")  String datasourceName,
            @Descriptor("Table name") @Argument("table") String tableName
            ) throws Exception {

        if (raw && csv) {
            throw new IllegalArgumentException("You must not simulatenously specify -r and -c, chooe either CSV or raw output.");
        }

        try (DataSourceContext ctx = new DataSourceContext(this.serviceRegistry, datasourceName);
             Connection conn = ctx.getDataSource().getConnection();
             Statement statement = conn.createStatement()) {

            if (limit >= 0) {
                statement.setMaxRows(limit);
            }

            DatabaseMetaData md = conn.getMetaData();

            String quote = md.getIdentifierQuoteString();

            String sql = "select * from " + quote + tableName + quote;
            runQuery(out,statement,sql,outputLength,csv,raw,encoding);
        }
    }

    @Descriptor("Execute SQL queries from stdin.")
    public void sql(PrintStream out,
            @Descriptor("Maximal output length of column data") @Parameter(names={"-l", "--length"}, absentValue = "32") int outputLength,
            @Descriptor("Maximal number of rows to show") @Parameter(names={"-L", "--limit"}, absentValue = "-1") int limit,
            @Descriptor("Output in CSV format") @Parameter(names={"-c", "--csv"}, presentValue="true", absentValue = "false") boolean csv,
            @Descriptor("Output in raw binary format") @Parameter(names={"-r", "--raw"}, presentValue="true", absentValue = "false") boolean raw,
            @Descriptor("Encoding of SQL script or raw output") @Parameter(names={"-e", "--encoding"}, absentValue = "UTF-8") String encoding,
            @Descriptor("Name of the registered datasource") @Argument("datasource")  String datasourceName
            ) throws Exception {

        try (DataSourceContext ctx = new DataSourceContext(this.serviceRegistry, datasourceName);
             Connection conn = ctx.getDataSource().getConnection();
             Statement statement = conn.createStatement();
             LineNumberReader reader = new LineNumberReader(new InputStreamReader(System.in,encoding))
             ) {

            if (limit >= 0) {
                statement.setMaxRows(limit);
            }

            StringBuilder sb = new StringBuilder();

            String line = reader.readLine();
            boolean inside_string = false;
            int nquery = 0;

            while (line != null) {

                if (!inside_string && (line.startsWith("--") || line.startsWith("#"))) {
                    line = reader.readLine();
                    continue;
                }

                int i0 = 0;

                for (int i=0;i<line.length();i = line.offsetByCodePoints(i,1)) {

                    int cp = line.codePointAt(i);

                    if (cp == ';' && !inside_string) {

                        sb.append(line.substring(i0,i));

                        String sql = sb.toString().trim();
                        sb.setLength(0);

                        if ("quit".equals(sql)) {
                            return;
                        }

                        if (nquery > 0) {
                            out.println();
                        }

                        ++nquery;

                        if (!raw && !csv) {
                            out.println(sql);
                        }

                        runQuery(out,statement,sql,outputLength,csv,raw,encoding);

                        i0 = line.offsetByCodePoints(i,1);
                    }
                    else if (cp == '\'') {
                        inside_string = !inside_string;
                    }
                }

                if (sb.length() > 0) {
                    sb.append(' ');
                }
                sb.append(line.substring(i0));
                line = reader.readLine();
            }

            if (sb.length() > 0) {

                String trailer = sb.toString().trim();

                if (!trailer.isEmpty()) {

                    if (inside_string) {

                        throw new IllegalArgumentException("Unterminated string in SQL query ["+trailer+"].");
                    }
                    else {
                        throw new IllegalArgumentException("Found trailing garbage ["+trailer+"] after last SQL statement.");
                    }
                }
            }
        }
    }
}
