package org.clazzes.svc.runner;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.clazzes.svc.api.ComponentManager;
import org.clazzes.svc.api.ConfigurationEngine;
import org.clazzes.svc.api.CoreService;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ServiceRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServiceContextImpl implements ServiceContext {

    private static final Logger log = LoggerFactory.getLogger(ServiceContextImpl.class);

    private final Map<Class<?>,Object> services;

    private final ConfigurationEngineImpl configurationEngine;
    private final ServiceRegistryImpl serviceRegistry;
    private final ComponentManagerImpl componentManager;

    // number of submitted manipulators
    private int submitted = 0;
    // number of running manipulators;
    private int counter = 0;

    public ServiceContextImpl() {

        this.services = new HashMap<Class<?>,Object>();
        this.services.put(CoreService.class,CoreServiceImpl.INSTANCE);

        this.configurationEngine = new ConfigurationEngineImpl();
        this.services.put(ConfigurationEngine.class,this.configurationEngine);

        this.serviceRegistry = new ServiceRegistryImpl();
        this.services.put(ServiceRegistry.class,this.serviceRegistry);

        this.componentManager = new ComponentManagerImpl();
        this.services.put(ComponentManager.class,this.componentManager);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> Optional<T> getService(Class<T> iface) {

        return Optional.ofNullable((T)this.services.get(iface));
    }

    public void start() {
        this.configurationEngine.start(this);
        this.serviceRegistry.start(this);
        this.componentManager.startBootLayers(this);
    }

    public void startForTest(InputStream configYaml, String configResource) throws IOException {
        this.configurationEngine.loadTestYaml(this,configYaml,configResource);
        this.serviceRegistry.start(this);
        this.componentManager.startBootLayers(this);
    }

    public void stop(int timeoutMillis) {
        this.componentManager.stopLayers(timeoutMillis);
        try {
            this.synchronize(timeoutMillis);
        } catch (InterruptedException e) {
            log.warn("Synchronization before serivce registry stop has been interrupted",e);
        }
        this.serviceRegistry.stop();
        this.configurationEngine.stop();
    }

    protected void manipulatorStarted() {

        int njobs;
        int nsubmitted;

        synchronized(this) {

            njobs = ++this.counter;
            nsubmitted = --this.submitted;
        }

        if (log.isDebugEnabled()) {
            log.debug("ServiceContextImpl.manipulatorStarted nsubmitted,njobs=[{},{}]",nsubmitted,njobs);
        }
    }

    protected void manipulatorStopped() {

        int njobs;

        synchronized(this) {
            njobs = --this.counter;

            if (log.isDebugEnabled()) {
                log.debug("ServiceContextImpl.manipulatorStopped njobs=[{}]",njobs);
            }

            if (njobs == 0) {
                this.notifyAll();
            }
        }
    }

    protected void mainpulatorScheduled() {

        int nsubmitted;
        synchronized(this) {
            nsubmitted = ++submitted;
        }

        if (log.isDebugEnabled()) {
            log.debug("ServiceContextImpl.manipulatorScheduled nsubmitted=[{}]",nsubmitted);
        }
    }
    /**
     * Wait fo the internal manipuilator counter to decrement.
     *
     * @param millis The number of milliseconds to wait.
     * @return The new job counter.
     * @throws InterruptedException If the current thread has been interrupted.
     */
    protected int waitForManipulatorDecrement(long millis) throws InterruptedException {

        int njobs;

        synchronized(this) {
            njobs = counter+submitted;

            if (njobs != 0) {
                this.wait(millis);
            }

            njobs = counter+submitted;
        }

        if (log.isDebugEnabled()) {
            log.debug("ServiceContextImpl.waitForManipulatorDecrement njobs=[{}]",njobs);
        }

        return njobs;
    }



    @Override
    public void synchronize(long timeoutMillis) throws InterruptedException {

        long n0 = System.nanoTime();
        long remaining = timeoutMillis;
        int njobs;

        do {

            njobs = this.waitForManipulatorDecrement(remaining);

            if (njobs == 0) {
                return;
            }

            remaining = (timeoutMillis * 1000000L + n0 - System.nanoTime() + 999999L) / 1000000L;

        } while(remaining > 0L);

        throw new IllegalStateException("Failed to synchronize internal jobs with ["+njobs+"] remaining after ["+timeoutMillis+"] milliseconds.");
    }

    @Override
    public void scheduleManipulator(Runnable runnable) {

        CoreServiceImpl.INSTANCE.getExecutorService().submit(()-> {

            try {
                this.manipulatorStarted();
                runnable.run();
            }
            finally {
                this.manipulatorStopped();
            }
        });
        this.mainpulatorScheduled();
    }

}
