/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner;

import java.security.MessageDigest;
import java.security.SecureRandom;
import java.util.Base64;
import java.util.Base64.Decoder;
import java.util.Base64.Encoder;
import java.util.HexFormat;
import java.util.Properties;

import javax.crypto.Cipher;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.SecretKeySpec;

public class SecretsStore {

    // 32 byte AES-256 masterkey
    private final byte[] masterKey;
    private final String kvc;
    private final Properties properties;

    private static final int GCM_IV_LENGTH = 12;
    private static final int GCM_TAG_LENGTH = 16;

    private static final String AES_GCM_ALGO = "AES-GCM";

    private static final SecureRandom secureRandom = new SecureRandom();

    private static String calcKVC(byte [] mk) {

        Cipher cipher;
        try {
            cipher = Cipher.getInstance("AES/ECB/NoPadding");

            SecretKeySpec cryptor = new SecretKeySpec(mk,"AES");

            cipher.init(Cipher.ENCRYPT_MODE,cryptor);

            byte[] enc = cipher.doFinal(new byte[16]);

            return HexFormat.of().withUpperCase().formatHex(enc,0,3);
        } catch (Exception e) {
            throw new IllegalArgumentException("Secrets store is unable to calculate KVC of 32 byte AES-256 master key.",e);
        }
}

    public SecretsStore(Properties properties, String masterKey) {
        super();

        Decoder dec = Base64.getDecoder();

        byte[] mk = dec.decode(masterKey);

        if (mk.length != 32) {
            throw new IllegalArgumentException("Secrets store must be initialized with a 32 byte AES-256 master key.");
        }

        this.properties = properties;
        this.masterKey = mk;
        this.kvc = calcKVC(mk);
    }

    public Properties getProperties() {
        return this.properties;
    }

    public String getKvc() {
        return this.kvc;
    }

    public byte[] diversifyMasterKey(String pid, String key) throws Exception {

        MessageDigest md = MessageDigest.getInstance("SHA-256");

        md.update(pid.getBytes("UTF-8"));
        md.update((byte)'\n');
        md.update(key.getBytes("UTF-8"));
        md.update((byte)'\n');

        byte[] diversifier = md.digest();

        for (int i=0;i<16;++i) {
            diversifier[i] ^= diversifier[i+16];
        }

        Cipher cipher = Cipher.getInstance("AES/ECB/NoPadding");

        SecretKeySpec mk = new SecretKeySpec(this.masterKey,"AES");

        cipher.init(Cipher.ENCRYPT_MODE,mk);

        return cipher.doFinal(diversifier,0,16);
    }

    public void encrypt(String pid, String key, String value) throws Exception {

        byte[] diversified = this.diversifyMasterKey(pid,key);

        Cipher cipher = Cipher.getInstance("AES/GCM/NoPadding");

        SecretKeySpec cryptor = new SecretKeySpec(diversified,"AES");

        byte[] iv = new byte[GCM_IV_LENGTH];

        secureRandom.nextBytes(iv);

        GCMParameterSpec gcmParameterSpec =
                new GCMParameterSpec(GCM_TAG_LENGTH*8,iv);

        cipher.init(Cipher.ENCRYPT_MODE,cryptor,gcmParameterSpec);

        byte[] encrypted = cipher.doFinal(value.getBytes("UTF-8"));

        Encoder b64Enc = Base64.getEncoder();

        String encryptedValue = "{"+AES_GCM_ALGO+","+b64Enc.encodeToString(iv)+"}" + b64Enc.encodeToString(encrypted);

        this.properties.setProperty(key,encryptedValue);
    }

    public String decrypt(String pid, String key) throws Exception {

        String encryptedValue = this.properties.getProperty(key);

        if (encryptedValue == null) {
            throw new IllegalArgumentException("There no value with key ["+key+"].");
        }

        if (!encryptedValue.startsWith("{")) {
            throw new IllegalArgumentException("Encyrpted value does not contain an encryption tag.");
        }

        int endOfTags = encryptedValue.indexOf('}',1);

        if (endOfTags < 0) {
            throw new IllegalArgumentException("Encyrpted value does not contain the end of an encryption tag.");
        }

        String[] tags = encryptedValue.substring(1,endOfTags).split(",");

        String algo = tags[0];

        if (!AES_GCM_ALGO.equals(algo)) {
            throw new IllegalArgumentException("Encyrpted value contains encryption tag with unknown algorithm ["+algo+"].");
        }

        if (tags.length != 2) {
            throw new IllegalArgumentException("Encyrpted value contains encryption tag for algorithm ["+algo+"] with wrong number of arguments.");
        }

        Decoder b64Dec = Base64.getDecoder();

        byte[] iv = b64Dec.decode(tags[1]);

        byte[] encrypted = b64Dec.decode(encryptedValue.substring(endOfTags+1));

        byte[] diversified = this.diversifyMasterKey(pid,key);

        Cipher cipher = Cipher.getInstance("AES/GCM/NoPadding");

        SecretKeySpec cryptor = new SecretKeySpec(diversified,"AES");

        GCMParameterSpec gcmParameterSpec =
                new GCMParameterSpec(GCM_TAG_LENGTH*8,iv);

        cipher.init(Cipher.DECRYPT_MODE,cryptor,gcmParameterSpec);

        byte[] decrypted = cipher.doFinal(encrypted);

        return new String(decrypted,"UTF-8");
    }

}
