
/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner;

import java.lang.ModuleLayer.Controller;
import java.lang.module.Configuration;
import java.lang.module.ModuleFinder;
import java.nio.file.Path;
import java.util.AbstractMap.SimpleEntry;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.clazzes.svc.api.Component;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ServicePriority;
import org.clazzes.svc.runner.opener.Opener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ComponentLayer {

    private static final Logger log = LoggerFactory.getLogger(ComponentLayer.class);

    private final LayerConfig layerConfig;
    private final ModuleLayer moduleLayer;
    private final String path;
    private final List<ComponentHolder> components;
    private final Map<Module,ComponentHolder> componentsByModule;

    protected ComponentLayer(LayerConfig layerConfig, ModuleLayer moduleLayer, String path, List<ComponentHolder> components) {
        this.layerConfig = layerConfig;
        this.moduleLayer = moduleLayer;
        this.path = path;
        this.components = components;
        this.componentsByModule = new HashMap<Module,ComponentHolder>();

        for (ComponentHolder component : components) {

            Module module = component.getComponent().getClass().getModule();

            ComponentHolder old = this.componentsByModule.put(module,component);

            if (old != null) {
                log.warn("Module [{}] defines more than one compnent apart from [{}]",
                         module,old);
            }
        }
    }

    protected static ComponentLayer resolve(LayerConfig layerConfig, ModuleLayer moduleLayer, String path) {

        for (OpensDirective open : layerConfig.getOpens()) {

            log.info("Precessing opens directive [{}]...",open);

            Optional<Module> src = moduleLayer.findModule(open.getSourceModule());

            if (src.isEmpty()) {
                log.warn("Cannot find source module [{}] in layer [{}]",
                            open.getSourceModule(),layerConfig.getKey());
                continue;
            }

            Optional<Module> target = moduleLayer.findModule(open.getTargetModule());

            if (target.isEmpty()) {
                log.warn("Cannot find target module [{}] in layer [{}]",
                            open.getTargetModule(),layerConfig.getKey());
                continue;
            }

            // Delgate opening of packages to the dedicated module
            // org.clazzes.svc.runner.opener,
            // so packages that need to be opened for sub-layers need
            // only be opened for org.clazzes.svc.runner.opener on the
            // command line like follows:
            //    --add-opens java.base/java.nio=org.clazzes.svc.runner.opener
            // This prevents us from opening central packages to
            // the whole svc-runner-core module.
            Opener.addOpens(src.get(),open.getSourcePackage(),target.get());

            log.info("Successfully opened package [{}] of [{}] to [{}].",
                    open.getSourcePackage(),src.get(),target.get());
        }

        ServiceLoader<Component> loader = ServiceLoader.load(moduleLayer,Component.class);

        List<ComponentHolder> components = new ArrayList<ComponentHolder>();

        loader.stream()
            // skip components from base layers, which have already been started
            // before.
            // This is, because ServiceLoader dives into all base layers.
            .filter(p -> moduleLayer.equals(p.type().getModule().getLayer()))
            .forEach(
                p -> {
                    Component component = p.get();
                    ServicePriority priority =
                        component.getClass().getAnnotation(ServicePriority.class);
                    ComponentHolder ch = new ComponentHolder(component,priority);
                    components.add(ch);
                    log.info("Resolved component [{}] with priority [{}].",
                             ch.getClassName(),
                             ch.getPriority());
                }
            );

        Collections.sort(components);

        return new ComponentLayer(layerConfig,moduleLayer,path,components);
    }

    /**
     * Create a component layer from an existing root layer.
     * @param moduleLayer The root layer.
     * @param path The already resolved path.
     * @return The component layer with resolved components.
     */
    public static ComponentLayer of(ModuleLayer moduleLayer, String path) {

        List<String> allNames = moduleLayer.configuration().modules().stream().map((m) -> m.name()).collect(Collectors.toList());

        Collections.sort(allNames);

        log.info("Creating boot layer for path [{}] with modules {}",path,allNames);

        return ComponentLayer.resolve(LayerConfig.ofBoot(),moduleLayer,path);
    }

    /**
     * Create a component layer from an existing parent layer
     * and all jar files from a given path.
     *
     * @param layerConfig The layer configuration POJO.
     * @param parent The parent layer.
     * @param path The already resolved path.
     * @return The component layer with resolved components.
     */
    public static ComponentLayer of(LayerConfig layerConfig, ModuleLayer parent, Path path) {

        ModuleFinder finder = ModuleFinder.of(path);

        if (log.isDebugEnabled()) {
            log.debug("Path [{}] contains modules [{}]",path,finder.findAll());
        }

        List<String> allNames = finder.findAll().stream().map(m -> m.descriptor().name()).collect(Collectors.toList());

        Collections.sort(allNames);

        log.info("Creating layer for path [{}] with modules {}",path,allNames);

        Configuration cf = parent.configuration().resolve(finder,ModuleFinder.of(),allNames);

        ClassLoader scl = ClassLoader.getSystemClassLoader();

        Controller nl = ModuleLayer.defineModulesWithOneLoader(cf,List.of(parent),scl);

        return ComponentLayer.resolve(layerConfig,nl.layer(),path.toString());
    }

    public String getParentLabel() {
        String ret = this.layerConfig.getParent();

        if (ret != null || this.layerConfig.isBoot()) {
            return ret;
        }
        else {
            return LayerConfig.BOOT_LABEL;
        }
    }

    public String getLabel() {
        return this.layerConfig.getLabel();
    }

    public String getKey() {
        return this.layerConfig.getKey();
    }

    public LayerConfig getLayerConfig() {
        return this.layerConfig;
    }

    public ModuleLayer getModuleLayer() {
        return this.moduleLayer;
    }

    public List<ComponentHolder> getComponents() {
        return this.components;
    }

    public ComponentHolder getByModule(Module module) {

        return this.componentsByModule.get(module);
    }

    public Stream<Map.Entry<String,ComponentHolder>> getComponentsByModuleName() {
        return this.componentsByModule.entrySet().stream().map(e ->
            new SimpleEntry<String,ComponentHolder>(e.getKey().getName(),e.getValue()));
    }

    public String getPath() {
        return this.path;
    }

    public static void startComponent(ServiceContext svcCtxt,
                                      ComponentHolder component) {

        svcCtxt.scheduleManipulator(() -> {

            try  {
                component.starting(Thread.currentThread());
            }
            catch(Throwable e) {
                log.error("Unable to initiate start of component [{}].",component);
                return;
            }

            try {
                log.info("Starting component [{}]...",component);
                component.getComponent().start(svcCtxt);
                component.started();
                log.info("Component [{}] started successfully.",component);
            }
            catch (Throwable e) {
                component.startFailed();
                component.error(log,"Starting component [{}] failed",component,e);
            }
        });
    }

    public void startAll(ServiceContext svcCtxt) {

        for (ComponentHolder component : this.components) {

            startComponent(svcCtxt,component);
        }
    }

    public static void stopComponent(ServiceContext svcCtxt,
                                     ComponentHolder component,
                                     long timeout) {
        svcCtxt.scheduleManipulator(() -> {

            try {
                component.stopTransition(timeout,
                        msg -> log.info(msg));

                component.stopping(Thread.currentThread());
            }
            catch(Throwable e) {
                log.error("Unable to initiate stop of component [{}]",component,e);
                return;
            }

            try {
                log.info("Stopping component [{}]...",component);
                component.getComponent().stop(svcCtxt);
                component.stopped();
                log.info("Component [{}] stopped successfully.",component);
            }
            catch (Throwable e) {
                component.stopFailed();
                component.error(log,"Stopping component [{}] failed",component,e);
            }
        });
    }

    public void stopAll(ServiceContext svcCtxt, long timeout) {

        for (int i = this.components.size();--i>=0;) {

            ComponentHolder component = this.components.get(i);

            stopComponent(svcCtxt,component,timeout);
        }
    }

}
