/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner;

import java.lang.module.ModuleDescriptor.Version;
import java.util.Arrays;
import java.util.Optional;
import java.util.function.Consumer;

import org.clazzes.svc.api.Component;
import org.clazzes.svc.api.ComponentState;
import org.clazzes.svc.api.ServicePriority;

public class ComponentHolder extends ThrowableBucket implements Comparable<ComponentHolder> {

    private final Component component;
    private final int priority;
    private ComponentState state;
    private Thread transitionThread;

    public ComponentHolder(Component component, int priority) {
        this.component = component;
        this.priority = priority;
        this.state = ComponentState.LOADED;
    }

    public ComponentHolder(Component component, ServicePriority priority) {
        this(component,
             priority != null ?
                priority.value() :
                ServicePriority.DEFAULT_PROIRITY);
    }

    public Component getComponent() {
        return this.component;
    }

    public String getClassName() {
        return this.component.getClass().getName();
    }

    public String getModuleName() {
        return this.component.getClass().getModule().getName();
    }

    public String getModuleNameAndVersion() {
        return this.component.getClass().getModule().getDescriptor().toNameAndVersion();
    }

    public Optional<Version> getModuleVersion() {
        return this.component.getClass().getModule().getDescriptor().version();
    }

    public int getPriority() {
        return this.priority;
    }

    public synchronized ComponentState getState() {
        return this.state;
    }

    protected void checkState(ComponentState target, ComponentState... allowed) {

        for (ComponentState a:allowed) {
            if (this.state == a) {
                this.state = target;
                return;
            }
        }

        throw new IllegalStateException("Cannot switch from state ["+this.state+
                                        "] to ["+target+"] (allowed source states are "+
                                        Arrays.toString(allowed)+").");
    }

    public synchronized void starting(Thread thread) {

        this.checkState(ComponentState.STARTING,
                        ComponentState.LOADED,ComponentState.STOPPED,ComponentState.STOP_FAILED);
        this.transitionThread = thread;
    }

    public synchronized void started() {

        // the compnent my have called commit() as part of the start method.
        if (this.state != ComponentState.COMMITTED) {
            this.checkState(ComponentState.STARTED,
                            ComponentState.STARTING);
        }
        this.transitionThread = null;
        this.notifyAll();
    }


    public synchronized void startFailed() {

        this.checkState(ComponentState.START_FAILED,
                        ComponentState.STARTING);
        this.transitionThread = null;
        this.notifyAll();
    }

    public synchronized void stopping(Thread thread) {

        this.checkState(ComponentState.STOPPING,
                        ComponentState.STARTED,
                        ComponentState.START_FAILED,
                        ComponentState.COMMITTED,
                        ComponentState.CONFIG_LISTENER_FAILED,
                        ComponentState.SERVICE_LISTENER_FAILED);
        this.state = ComponentState.STOPPING;
        this.transitionThread = thread;
    }

    public synchronized void stopped() {

        this.checkState(ComponentState.STOPPED,
                        ComponentState.STOPPING);
        this.transitionThread = null;
        this.notifyAll();
    }

    public synchronized void stopFailed() {

        this.checkState(ComponentState.STOP_FAILED,
                        ComponentState.STOPPING);
        this.transitionThread = null;
        this.notifyAll();
    }

    public synchronized ComponentState stopTransition(long timeout,
                        Consumer<String> callback) throws InterruptedException {

        if (this.transitionThread == null) {
            return this.state;
        }

        callback.accept("Stopping transition on ["+this+"].");
        this.transitionThread.interrupt();
        this.wait(timeout);
        callback.accept("Successfully stopped transition on ["+this+"].");
        return this.state;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((component == null) ? 0 : component.hashCode());
        result = prime * result + priority;
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        ComponentHolder other = (ComponentHolder) obj;
        if (component == null) {
            if (other.component != null)
                return false;
        } else if (!component.equals(other.component))
            return false;
        if (priority != other.priority)
            return false;
        return true;
    }

    public static int compareVersion(Optional<Version> a, Optional<Version> b) {

        if (a.isPresent()) {
            if (b.isPresent()) {
                return a.get().compareTo(b.get());
            }
            else {
                return 1;
            }
        }
        else {
            return b.isPresent() ? -1 : 0;
        }
}

    @Override
    public int compareTo(ComponentHolder o) {

        if (this.priority > o.getPriority()) {
            return 1;
        }
        else if (this.priority < o.getPriority()) {
            return -1;
        }
        else {

            int ret = this.getModuleName().compareTo(o.getModuleName());

            if (ret != 0) {
                return ret;
            }

            ret = compareVersion(this.getModuleVersion(),o.getModuleVersion());

            if (ret != 0) {
                return ret;
            }

            return this.getClassName().compareTo(o.getClassName());
        }
    }

    @Override
    public String toString() {
        return "ComponentHolder [component=" + this.getModuleNameAndVersion() +
                    "/" + this.getClassName() +
                    ", priority=" + this.priority +
                    ", state=" + this.getState() +
                    "]";
    }


}
