/***********************************************************
 *
 * Service API of the clazzes.org project
 * https://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.api.monitoring;

public enum HealthStatus {

    /**
     * Critical Status, intervention is required to bring system back to life.
     */
    CRITICAL(0.0),
    /**
     * Warning, attention required but system is operational.
     */
    WARN(0.5),
    /**
     * System is fully operational.
     */
    OK(1.0),
    /**
     * <p>Health Check itself did not execute properly (no reliable status is known).</p>
     *
     * <p>Most likely, some underlying infrastructure like DB conenctions or the like
     * is not properly set up.</p>
     */
    HEALTH_CHECK_ERROR(Double.NaN);

    private final double value;

    private HealthStatus(double value) {
        this.value = value;
    }

    /**
     * @param v The value <code>0.0</code>, <code>0.5</code>, <code>1.0</code>
     *              or <code>NaN</code> of the HealthStatus to return.
     * @return The corresponding HealthStatus enum value.
     * @throws IllegalArgumentException if the given value does not correspond
     *              to a health status.
     */
    public static HealthStatus ofValue(double v) {

        if (v == 1.0) {
            return OK;
        }
        else if (v == 0.0) {
            return CRITICAL;
        }
        else if (v == 0.5) {
            return WARN;
        }
        else if (Double.isNaN(v)) {
            return HEALTH_CHECK_ERROR;
        }
        else {
            throw new IllegalArgumentException(
                        "Unsupported double value ["+v+"] for HealthStatus.");
        }
    }

    /**
     * @return The value of this status as a double-valued gauge.
     */
    public double getValue() {
        return this.value;
    }

}
