/***********************************************************
 *
 * Service API of the clazzes.org project
 * https://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.api.monitoring;

import java.util.ArrayList;
import java.util.List;

/**
 * <p>A <code>HealthInfo</code> which is generated from a history.</p>
 */
public class HealthInfoWithLog extends HealthInfo {

    private final List<HealthInfo> log;

    /**
     * Construct a health info for a status with an <code>null</code>
     * explanation and no exception information.
     * @param status The health status to set.
     */
    public HealthInfoWithLog(HealthStatus status, List<HealthInfo> log) {

        super(status);
        this.log = log;
    }

    /**
     * Construct a health info with an explanation and no exception information.
     * @param status The health status to set.
     * @param explanation The explanation to set.
     */
    public HealthInfoWithLog(HealthStatus status, String explanation, List<HealthInfo> log) {
        super(status,explanation);
        this.log = log;
    }

    /**
     * Construct a health info with an explanation and exception information.
     * @param status The health status to set.
     * @param explanation The explanation to set.
     */
    public HealthInfoWithLog(HealthStatus status, String explanation, Throwable exception, List<HealthInfo> log) {
        super(status,explanation,exception);
        this.log = log;
    }

    /**
     * <p>Construct a HealthInfoWithLog from a given log of health infos.
     * The explanation of the generated health info is a summary of the
     * health infos with their statii.
     * </p>
     *
     * <p>It is not recommended to add health infos with a status of
     * {@link HealthStatus#HEALTH_CHECK_ERROR} to the log, because
     * this status will be overruled by all other statii and it is reserved
     * HealthChecks, which throw an exception in their <code>call()</code>
     * method.
     * </p>
     *
     * @param title The title in the generated explanation.
     * @param log The log of HealthInfos.
     * @return A health info with the most critical status found in the log.
     */
    public static HealthInfoWithLog ofLog(String title, List<HealthInfo> log) {

        return HealthInfoWithLog.ofLogAndTheInfo(title,log,null);
    }

    /**
     * <p>Construct a HealthInfoWithLog from a given log of health infos.
     * The explanation of the generated health info is a summary of the
     * health infos with their statii and the message of the given exception.
     * </p>
     *
     * <p>It is not recommended to add health infos with a status of
     * {@link HealthStatus#HEALTH_CHECK_ERROR} to the log, because
     * this status will be overruled by all other statii and it is reserved
     * HealthChecks, which throw an exception in their <code>call()</code>
     * method.
     * </p>
     *
     * @param title The title in the generated explanation.
     * @param log The log of HealthInfos.
     * @param e The execption to be added to the final log and the explanation of the
     *
     * @return A health info with the most critical status found in the log and the given
     *         exception.
     */
    public static HealthInfoWithLog ofLog(String title, List<HealthInfo> log, Throwable e) {

        HealthInfo theInfo = HealthInfo.ofException(e);

        List<HealthInfo> realLog = new ArrayList<HealthInfo>(log.size()+1);

        realLog.addAll(log);
        realLog.add(theInfo);

        return HealthInfoWithLog.ofLogAndTheInfo(title,realLog,theInfo);
    }
    /**
     * Internal function to construct a health info with a log.
     *
     * @param title The title in the generated explanation.
     * @param log The log of HealthInfos.
     * @param theInfo top info, the single top error, e.g. the top critical error.
     *            If <code>null</code>, the top erro is auto-detected.
     * @return A health info with a log.
     */
    protected static HealthInfoWithLog ofLogAndTheInfo(String title, List<HealthInfo> log, HealthInfo theInfo) {

        HealthStatus status;
        String explanation;

        if (log == null || log.size() == 0) {

            explanation = "No health information log available.";
            status = HealthStatus.HEALTH_CHECK_ERROR;
        }
        else {

            HealthStatus[] statii = HealthStatus.values();

            int[] counts = new int[statii.length];
            // if for a status a single info is found, record it here, if the
            // single top info is not passed in.
            HealthInfo[] topInfos = theInfo == null ? new HealthInfo[statii.length] : null;

            for (HealthInfo info : log) {

                int i = info.getStatus().ordinal();

                if (++counts[i] == 1 && topInfos != null) {
                    topInfos[i] = info;
                }
            }

            status = null;

            StringBuilder sb = new StringBuilder();

            sb.append(title);
            sb.append(" [");

            for (int i=0;i<statii.length;++i) {

                int n = counts[i];

                if (n > 0) {

                    if (status == null) {
                        status = statii[i];

                        if (theInfo == null && n == 1) {
                            theInfo = topInfos[i];
                        }
                    }
                    else {
                        sb.append('/');
                    }

                    sb.append(n);
                    sb.append(' ');
                    sb.append(statii[i]);
                }
            }

            sb.append(']');

            if (theInfo != null) {
                sb.append(": ");
                sb.append(theInfo.getExplanation());
            }

            explanation = sb.toString();
        }

        return new HealthInfoWithLog(status,explanation,
            theInfo == null ? null : theInfo.getException(),log);
    }

    /**
     * @return The log, which generated this health info.
     */
    public List<HealthInfo> getLog() {
        return this.log;
    }

    @Override
    public String toString() {
        return "HealthInfoWithLog [status=" + this.getStatus() +
                    ", explanation=" + this.getExplanation() + "]";
    }

}
