/***********************************************************
 *
 * Service API of the clazzes.org project
 * https://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.api.monitoring;

import java.util.Objects;

/**
 * <p>A <code>HealthStatus</code> with an optional explanation.</p>
 *
 * <p>Comparision is delegated to the supplied <code>HealthStatus</code>.</p>
 */
public class HealthInfo implements Comparable<HealthInfo> {

    private final HealthStatus status;
    private final String explanation;
    private final Throwable exception;

    /**
     * Construct a health info for a status with a <code>null</code>
     * explanation and no exception information.
     * @param status The health status to set.
     */
    public HealthInfo(HealthStatus status) {

        this.status = Objects.requireNonNull(status);
        this.explanation = null;
        this.exception = null;
    }

    /**
     * Construct a health info with an explanation and no exception information.
     * @param status The health status to set.
     * @param explanation The explanation to set.
     */
    public HealthInfo(HealthStatus status, String explanation) {
        this.status = Objects.requireNonNull(status);
        this.explanation = explanation;
        this.exception = null;
    }

    /**
     * Construct a health info with an explanation and exception information.
     * @param status The health status to set.
     * @param explanation The explanation to set.
     */
    public HealthInfo(HealthStatus status, String explanation, Throwable exception) {
        this.status = Objects.requireNonNull(status);
        this.explanation = explanation;
        this.exception = exception;
    }

    /**
     * @param v The value <code>0.0</code>, <code>0.5</code>, <code>1.0</code>
     *              or <code>NaN</code> of the HealthStatus to return.
     * @return The corresponding HealthInfo with a <code>null</code> explanation.
     * @throws IllegalArgumentException if the given value does not correspond
     *              to a health status.
     */
    public static HealthInfo ofValue(double v) {
        return new HealthInfo(HealthStatus.ofValue(v));
    }

    /**
     * @param v The value <code>0.0</code>, <code>0.5</code>, <code>1.0</code>
     *              or <code>NaN</code> of the HealthStatus to return.
     * @param explanation The explanation to set.
     * @return The corresponding HealthInfo with a <code>null</code> explanation.
     * @throws IllegalArgumentException if the given value does not correspond
     *              to a health status.
     */
    public static HealthInfo ofValue(double v, String explanation) {
        return new HealthInfo(HealthStatus.ofValue(v),explanation);
    }

    /**
     * <p>Construct a health info from an exception.
     * </p>
     * <p>The status is <code>CRITICAL</code> for instances of
     * <code>Exception</code> or <code>HEALTH_CHECK_ERROR</code>,
     * if <code>e</code> is an instance of <code>Error</code>.
     * </p>
     *
     * @param e The exception.
     * @return A health info with status {@link HealthStatus#CRITICAL}
     *         or {@link HealthStatus#HEALTH_CHECK_ERROR}.
     */
    public static HealthInfo ofException(Throwable e) {
        return new HealthInfo(
                e instanceof Error ?
                    HealthStatus.HEALTH_CHECK_ERROR :
                    HealthStatus.CRITICAL,
                e.toString(),e);
    }

    public HealthStatus getStatus() {
        return this.status;
    }

    public String getExplanation() {
        return this.explanation;
    }

    public Throwable getException() {
        return this.exception;
    }

    @Override
    public int compareTo(HealthInfo o) {

        return this.status.compareTo(o.status);
    }

    @Override
    public String toString() {
        return "HealthInfo [status=" + this.status +
                    ", explanation=" + this.explanation + "]";
    }

}
