/***********************************************************
 *
 * Service API of the clazzes.org project
 * https://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.api;

import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

/**
 * A service registry used for registering dynamic services.
 */
public interface ServiceRegistry {

    /**
     * <p>Add a service to the registry for a given key.
     * </p>
     *
     * @param <T> interface type parameter.
     * @param key The registry key of the service.
     * @param iface The interface class of the service.
     * @param service The service implementation to add.
     */
    public <T> void addService(String key, Class<T> iface, T service);

    /**
     * <p>Remove a service from the registry and a given key.
     * </p>
     * <p>If the given service implementation is not registered
     * under the given key, the method returns <code>false</code>
     * and logs a likewise warning.
     * </p>
     *
     * @param <T> interface type parameter.
     * @param key The registry key of the service.
     * @param iface The service implementation to remove.
     * @return Whether the service has been previoulsy registered
     *         and removed.
     */
    public <T> boolean removeService(String key, Class<T> iface);

    /**
     * <p>Get a service for a given interface and key.
     * </p>
     * <p>This method never returns <code>null</code> and calling
     * <code>getService(key,iface).get()</code> throws
     * {@link NoSuchElementException}, if no such service is known.
     * </p>
     *
     * @param <T> interface type parameter.
     * @param key The registry key of the service.
     * @param iface The interface class of the service.
     * @return An optional instance of the service for the given
     *         interface and key.
     */
    public <T> Optional<T> getService(String key, Class<T> iface);

    /**
     * <p>Get all services registered for a given interface.
     * </p>
     *
     * @param <T> interface type parameter.
     * @param iface The interface class of the service.
     * @return A read-only view of the currently registered services for the
     *         given interface indexed by registry keys.
     */
    public <T> Map<String,T> getAll(Class<T> iface);

    /**
     * List all known services for informational purposes.
     * @return A list of all registered services.
     */
    public List<ServiceInfo> listServices();

    /**
     * List all known services of agivern interface for
     * informational purposes.
     * @return A list of all registered services for the given interface.
     */
    public List<ServiceInfo> listServices(String iface);

    /**
     * <p>Get an arbitrary service for a given interface.
     * </p>
     * <p>This method never returns <code>null</code> and calling
     * <code>getAny(iface).get()</code> throws
     * {@link NoSuchElementException}, if no such service is known.
     * </p>
     *
     * @param <T> interface type parameter.
     * @param iface The interface class of the service.
     * @return An optional instance of the service for the given
     *         interface and key.
     */
    public <T> Optional<T> getAny(Class<T> iface);

    /**
     * <p>Listen for a service instance of a given interface and key
     * to become available.
     * </p>
     *
     * @param <T> interface type parameter.
     * @param key The key of the service to listen for.
     * @param iface The interface class of the service to listen for.
     * @param added Callback when the service becomes available.
     * @param removed Callback when the service disappears.
     * @return An autocloseable, which ends the subscription if closed.
     */
    public <T> AutoCloseable listen(String key, Class<T> iface,
                                    Consumer<T> added,
                                    Consumer<T> removed);

    /**
     * <p>Listen for any service instance of a given interface to
     * become available.
     * </p>
     *
     * @param <T> interface type parameter.
     * @param iface The interface class of the service to listen for.
     * @param added Callback when any key/service becomes available.
     * @param removed Callback when any key/service disappears.
     * @return An autocloseable, which ends the subscription if closed.
     */
    public <T> AutoCloseable listenAll(Class<T> iface,
                                       BiConsumer<String,T> added,
                                       BiConsumer<String,T> removed);
}
