/***********************************************************
 *
 * Service API of the clazzes.org project
 * https://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.api;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

/**
 * <p>A proxy factory, which creates proxies to call the currently
 * registered service in a service registry for a given key and interface.
 * </p>
 */
public abstract class ServiceProxyFactory {

    private static final class ServiceInvocationHandler<T> implements InvocationHandler {
        private final ServiceRegistry registry;
        private final String key;
        private final Class<T> iface;

        private ServiceInvocationHandler(ServiceRegistry registry, String key, Class<T> iface) {
            this.registry = registry;
            this.key = key;
            this.iface = iface;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            T t = registry.getService(key,iface).get();

            try {
                return method.invoke(t,args);
            }
            catch (InvocationTargetException ite) {
                throw ite.getCause();
            }
        }

        @Override
        public String toString() {
            return "ServiceProxy["+this.key+","+this.iface.getName()+"]";
        }
    }

    private static final class AnyServiceInvocationHandler<T> implements InvocationHandler {
        private final ServiceRegistry registry;
        private final Class<T> iface;

        private AnyServiceInvocationHandler(ServiceRegistry registry, Class<T> iface) {
            this.registry = registry;
            this.iface = iface;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            T t = registry.getAny(iface).get();

            try {
                return method.invoke(t,args);
            }
            catch (InvocationTargetException ite) {
                throw ite.getCause();
            }
        }

        @Override
        public String toString() {
            return "AnyServiceProxy["+this.iface.getName()+"]";
        }
    }

    /**
     * <p>Create a proxy, which calls the delegate fetched by</p>
     * <pre>
     * registry.getService(key,iface).get();
     * </pre>
     *
     * @param <T> The inteface type parameter.
     * @param registry The service registry to invoke for the service.
     * @param key The service key.
     * @param iface The interface class.
     * @return A proxy, which calls the registered service
     *         in the service registry.
     */
    @SuppressWarnings("unchecked")
    public static final <T> T getServiceProxy(ServiceRegistry registry, String key, Class<T> iface) {

        return (T)Proxy.newProxyInstance(
                iface.getClassLoader(),
                new Class<?>[] {
                    iface
                },
                new ServiceInvocationHandler<T>(registry,key,iface));
    }

    /**
     * <p>Create a proxy, which calls the delegate fetched by</p>
     * <pre>
     * registry.getAny(iface).get();
     * </pre>
     *
     * @param <T> The inteface type parameter.
     * @param registry The service registry to invoke for the service.
     * @param iface The interface class.
     * @return A proxy, which calls any registered service
     *         of the given interface in the service registry.
     */
    @SuppressWarnings("unchecked")
    public static final <T> T getServiceProxy(ServiceRegistry registry, Class<T> iface) {

        return (T)Proxy.newProxyInstance(
                iface.getClassLoader(),
                new Class<?>[] {
                    iface
                },
                new AnyServiceInvocationHandler<T>(registry,iface));
    }
}
