/***********************************************************
 *
 * Service API of the clazzes.org project
 * https://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.api;

import java.util.NoSuchElementException;
import java.util.Optional;

/**
 * A service context, which is a singleton in a server process
 * or an AutoCloseable context for Tests.
 */
public interface ServiceContext {

    /**
     * <p>Get a service for a given interface provided for the given
     * context.
     * </p>
     * <p>This method never returns <code>null</code> and calling
     * <code>getService(iface).get()</code> throws
     * {@link NoSuchElementException}, if no such service is known.
     * </p>
     * <p>If multiple services are registered for the given interface,
     * one unpredictable instance is returned.</p>
     * <p>The list of interfaces used to retrieve here are:
     * <ul>
     * <li><code>CoreService</code> returned as the global singleton.</li>
     * <li><code>ConfigurationEngine</code> as per-context instance.</li>
     * <li><code>ServiceRegistry</code> as per-context instance.</li>
     * <li><code>ComponentManager</code> as per-context instance.</li>
     * </ul>
     *
     * @param <T> interface type parameter.
     * @param iface The interface class of the service.
     * @return An optional instance of the service for the given
     *         interface.
     */
    public <T> Optional<T> getService(Class<T> iface);

    /**
     * Schedule a manipulator runnable, which alters the state of this
     * context.
     * @param runnable A runnable, which will manipulate the state of this
     *          context.
     */
    public void scheduleManipulator(Runnable runnable);

    /**
     * Wait for all internal listeners, start and stop jobs started by
     * {@link #scheduleManipulator(Runnable)} to be finished.
     * This is usually needed to start running unit tests on top of an
     * ad hoc context.
     * @param timeoutMillis The overall timeout in milliseconds.
     * @throws InterruptedException When the current thread has been interrupted.
     */
    public void synchronize(long timeoutMillis) throws InterruptedException;

}
