/***********************************************************
 *
 * Service API of the clazzes.org project
 * https://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.api;

import java.io.Serializable;
import java.lang.module.ModuleDescriptor.Version;
import java.net.URI;
import java.util.Optional;

/**
 * An information on a deployed java module.
 */
public class ModuleInfo implements Serializable, Comparable<ModuleInfo> {

    private static final long serialVersionUID = -2011953965659972023L;

    private final String layer;
    private final String name;
    private final Optional<Version> version;
    private final Optional<URI> location;

    /**
     *
     * @param layer The layer label, which is <code>$boot</code> for the boot layer.
     * @param name The name of the module.
     * @param version The optional version of the module.
     * @param location The location of the jar file of the module.
     */
    public ModuleInfo(String layer, String name, Optional<Version> version, Optional<URI> location) {
        this.layer = layer;
        this.name = name;
        this.version = version;
        this.location = location;
    }

    public String getLayer() {
        return this.layer;
    }

    public String getName() {
        return this.name;
    }

    public Optional<Version> getVersion() {
        return this.version;
    }

    public Optional<URI> getLocation() {
        return this.location;
    }

    @Override
    public int compareTo(ModuleInfo o) {

        String lo = o.getLayer();

        if (this.layer == null) {
            return lo == null ? 0 : -1;
        }
        else if (lo == null) {
            return 1;
        }

        int ret = this.layer.compareTo(lo);

        if (ret != 0) {
            return ret;
        }

        String no = o.getName();

        if (this.name == null) {
            return no == null ? 0 : -1;
        }
        else if (no == null) {
            return 1;
        }

        ret = this.name.compareTo(no);

        if (ret != 0) {
            return ret;
        }

        Optional<Version> vo = o.getVersion();

        if (this.version.isEmpty()) {
            return vo.isEmpty() ? 0 : -1;
        }
        else if (vo.isEmpty()) {
            return 1;
        }

        return this.version.get().compareTo(vo.get());
    }

}
