/***********************************************************
 *
 * Service API of the clazzes.org project
 * https://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.api;

import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * Helper methods for accessing structured configuration sets.
 */
public abstract class ConfigurationHelper {

    /**
     * Convert an object from a parsed configuration to a String.
     * @param o An object, which might be <code>null</code>, a
     *          String, Number or Boolean or a {@link Supplier}
     *          of one of these object types.
     * @return The extracted String or <code>null</code>.
     */
    public static String extractString(Object o) {

        if (o == null) {
            return null;
        }
        else if (o instanceof String s) {
            return s;
        }
        else if (o instanceof Supplier supp) {

            return extractString(supp.get());
        }
        else if ((o instanceof Number) || (o instanceof Boolean)) {
            return o.toString();
        }
        else {
            throw new IllegalArgumentException("Cannot convert object of type ["+o.getClass()+"] to String");
        }
    }

    /**
     * Convert an object from a parsed configuration to an Integer.
     * @param o An object, which might be <code>null</code>, a
     *          String or Number or a {@link Supplier}
     *          of one of these object types.
     * @return The extracted Integer or <code>null</code>.
     */
    public static Integer extractInteger(Object o) {

        if (o == null) {
            return null;
        }
        else if (o instanceof Integer) {
            return (Integer)o;
        }
        else if (o instanceof Number n) {

            if (n.longValue() != n.intValue()) {
                throw new IllegalArgumentException("Long value ["+n+"] is out of range for Integer");
            }

            return n.intValue();
        }
        else if (o instanceof Supplier supp) {

            return extractInteger(supp.get());
        }
        else if (o instanceof String s) {
            return Integer.valueOf(s);
        }
        else {
            throw new IllegalArgumentException("Cannot convert object of type ["+o.getClass()+"] to Integer");
        }
    }

    /**
     * Convert an object from a parsed configuration to a Long.
     * @param o An object, which might be <code>null</code>, a
     *          String or Number or a {@link Supplier}
     *          of one of these object types.
     * @return The extracted Long or <code>null</code>.
     */
    public static Long extractLong(Object o) {

        if (o == null) {
            return null;
        }
        else if (o instanceof Long) {
            return (Long)o;
        }
        else if (o instanceof Number n) {
            return n.longValue();
        }
        else if (o instanceof Supplier supp) {

            return extractLong(supp.get());
        }
        else if (o instanceof String s) {
            return Long.valueOf(s);
        }
        else {
            throw new IllegalArgumentException("Cannot convert object of type ["+o.getClass()+"] to Long");
        }
    }

    /**
     * Convert an object from a parsed configuration to a Double.
     * @param o An object, which might be <code>null</code>, a
     *          String or Number or a {@link Supplier}
     *          of one of these object types.
     * @return The extracted Double or <code>null</code>.
     */
    public static Double extractDouble(Object o) {

        if (o == null) {
            return null;
        }
        else if (o instanceof Double) {
            return (Double)o;
        }
        else if (o instanceof Number n) {
            return n.doubleValue();
        }
        else if (o instanceof Supplier supp) {

            return extractDouble(supp.get());
        }
        else if (o instanceof String s) {
            return Double.valueOf(s);
        }
        else {
            throw new IllegalArgumentException("Cannot convert object of type ["+o.getClass()+"] to Double");
        }
    }

    /**
     * Convert an object from a parsed configuration to a Boolean.
     * Accepted strings are "true" or "false", which are matched case-insensitive.
     * @param o An object, which might be <code>null</code>, a
     *          String or Number or a {@link Supplier}
     *          of one of these object types.
     * @return The extracted Boolean or <code>null</code>.
     */
    public static Boolean extractBoolean(Object o) {

        if (o == null) {
            return null;
        }
        else if (o instanceof Boolean) {
            return (Boolean)o;
        }
        else if (o instanceof Supplier supp) {

            return extractBoolean(supp.get());
        }
        else if (o instanceof String s) {

            if ("true".equalsIgnoreCase(s)) {
                return Boolean.TRUE;
            }
            else if ("false".equalsIgnoreCase(s)) {
                return Boolean.FALSE;
            }
            else {
                throw new IllegalArgumentException("String ["+s+"] cannot be converted to boolean.");
            }

        }
        else {
            throw new IllegalArgumentException("Cannot convert object of type ["+o.getClass()+"] to Integer");
        }
    }

    @SuppressWarnings("unchecked")
    private static Object getSubItem(Map<String,?> config, String path) {

        Map<String,?> current = config;

        int i=1;
        int ipos = 0,npos;

        while ((npos = path.indexOf('.',ipos)) >= 0) {

            String component = path.substring(ipos,npos);

            if (component.isBlank()) {
                throw new IllegalArgumentException("Empty path component ["+i+"] in path ["+path+"]");
            }

            Object sub = current.get(component);

            if (sub instanceof Map map) {
                current = map;
            }
            else if (sub instanceof List lst) {
                current = listToMap(lst);
            }
            else {
                throw new IllegalArgumentException("Non-map in subpath ["+path.substring(0,npos)+"] of path ["+path+"]");
            }

            ipos = npos + 1;
            ++i;
        }

        String component = path.substring(ipos);

        if (component.isBlank()) {
            throw new IllegalArgumentException("Empty trailing path component in path ["+path+"]");
        }

        return current.get(component);
    }

    protected static final Map<String,?> listToMap(List<?> lst) {

        Map<String,Object> ret = new TreeMap<String,Object>();
        for (int i=0;i<lst.size();++i) {
            ret.put(String.valueOf(i+1),lst.get(i));
        }
        return ret;
    }

    /**
     * Extract a subtree from a configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The subtree or <code>null</code>, if no such subtree exists.
     */
    @SuppressWarnings("unchecked")
    public static Map<String,?> getSubTree(Map<String,?> config, String path) {

        Object sub = getSubItem(config,path);

        if (sub == null) {
            return null;
        }
        else if (sub instanceof Map map) {
            return map;
        }
        else if (sub instanceof List lst) {
            return listToMap(lst);
        }
        else {
            throw new IllegalArgumentException("Non-map at config path ["+path+"]");
        }
    }

    protected static List<?> getList(Map<String,?> config, String path) {

        Object sub = getSubItem(config,path);

        if (sub == null) {
            return null;
        }
        else if (sub instanceof List lst) {
            return lst;
        }
        else {
            throw new IllegalArgumentException("Non-list at config path ["+path+"]");
        }
    }

    /**
     * Extract a mandatory subtree from a configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The subtree.
     * @throws IllegalArgumentException if the subtree does not exist.
     */
    public static Map<String,?> getMandatorySubTree(Map<String,?> config, String path) {

        Map<String,?> sub = getSubTree(config,path);

        if (sub == null) {
            throw new IllegalArgumentException("Sub-tree ["+path+"] does not exist.");
        }

        return sub;
    }

    /**
     * Extract a string from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The string or <code>null</code> if nothing is found under
     *         the given path.
     */
    public static String getString(Map<String,?> config, String path) {

        Object sub = getSubItem(config,path);

        try {
            return extractString(sub);
        }
        catch(Exception e) {
            throw new IllegalArgumentException("Config value ["+path+"] is not a string",e);
        }
    }

    /**
     * Extract a mandatory string from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The string.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static String getMandatoryString(Map<String,?> config, String path) {

        String s = getString(config,path);

        if (s == null || s.isBlank()) {
            throw new IllegalArgumentException("Config value ["+path+"] has not been specified.");
        }

        return s;
    }

    /**
     * Extract a mandatory string from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @param defaultValue A default value, if the given path does not exist or the value
     *             is blank.
     * @return The string.
     */
    public static String getString(Map<String,?> config, String path, String defaultValue) {

        String s = getString(config,path);

        if (s == null || s.isBlank()) {
            return defaultValue;
        }

        return s;
    }

    /**
     * Get a list of strings from the given path, which contains a YAML list.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return A list of strings or <code>null</code> if nothing is found under
     *         the given path.
     */
    public static List<String> getStringList(Map<String,?> config, String path) {

        List<?> lst = getList(config,path);

        if (lst == null) {
            return null;
        }

        return lst.stream().map((x) -> extractString(x)).collect(Collectors.toList());
    }

    /**
     * Get a list of strings from the given path, which contains a YAML list.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return A list of strings.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static List<String> getMandatoryStringList(Map<String,?> config, String path) {

        List<String> ret = getStringList(config,path);

        if (ret == null) {
            throw new IllegalArgumentException("Config value ["+path+"] has not been specified.");
        }

        return ret;
    }

    /**
     * Extract an enum value from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param cls The enum class to convert the value to.
     * @param path A path, which might be dot-separated.
     * @return The enum value or <code>null</code> if nothing is found under
     *         the given path.
     */
    public static <T extends Enum<T>> T getEnum(Map<String,?> config, Class<T> cls, String path) {

        Object sub = getSubItem(config,path);

        try {

            String v = extractString(sub);

            if (v == null) {
                return null;
            }
            else {
                return Enum.valueOf(cls,v);
            }
        }
        catch(Exception e) {
            throw new IllegalArgumentException("Config value ["+path+"] is not a enum of type ["+cls.getSimpleName()+"]",e);
        }
    }

    /**
     * Extract a mandatory enum value from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param cls The enum class to convert the value to.
     * @param path A path, which might be dot-separated.
     * @return The enum value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static <T extends Enum<T>> T getMandatoryEnum(Map<String,?> config, Class<T> cls, String path) {

        T t = getEnum(config,cls,path);

        if (t == null) {
            throw new IllegalArgumentException("Config value ["+path+"] has not been specified.");
        }

        return t;
    }

    /**
     * Extract a mandatory enum value from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param cls The enum class to convert the value to.
     * @param path A path, which might be dot-separated.
     * @param defaultValue The default value, if the given path is not defined.
     * @return The enum value.
     */
    public static <T extends Enum<T>> T getEnum(Map<String,?> config, Class<T> cls, String path, T defaultValue) {

        T t = getEnum(config,cls,path);

        if (t == null) {
            return defaultValue;
        }

        return t;
    }

    /**
     * Get a set of enum values from the given path, which contains a YAML list.
     * @param config a parsed, structured configuration.
     * @param cls The enum class to convert the value to.
     * @param path A path, which might be dot-separated.
     * @return A set of enum values or <code>null</code> if nothing is found under
     *         the given path.
     */
    public static <T extends Enum<T>> EnumSet<T> getEnumSet(Map<String,?> config, Class<T> cls, String path) {

        List<?> lst = getList(config,path);

        if (lst == null) {
            return null;
        }

        EnumSet<T> ret = EnumSet.noneOf(cls);

        lst.stream().forEach((x) -> ret.add(Enum.valueOf(cls,extractString(x))));

        return ret;
    }

    /**
     * Get a set of enum values from the given path, which contains a YAML list.
     * @param config a parsed, structured configuration.
     * @param cls The enum class to convert the value to.
     * @param path A path, which might be dot-separated.
     * @return A set of enum values.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static <T extends Enum<T>> EnumSet<T> getMandatoryEnumSet(Map<String,?> config, Class<T> cls, String path) {

        EnumSet<T> ret = getEnumSet(config,cls,path);

        if (ret == null) {
            throw new IllegalArgumentException("Config value ["+path+"] has not been specified.");
        }

        return ret;
    }

    /**
     * Extract an object parsed from a string out of a structured configuration.
     * @param config a parsed, structured configuration.
     * @param factory A factory method for creating instances like
     *        <code>URI::create</code> or <code>OffsetDateTime::parse</code>.
     * @param path A path, which might be dot-separated.
     * @return The parsed value or <code>null</code> if nothing is found under
     *         the given path.
     */
    public static <T> T getParsed(Map<String,?> config, Function<? super String,T> factory, String path) {

        Object sub = getSubItem(config,path);

        try {

            String v = extractString(sub);

            if (v == null) {
                return null;
            }
            else {
                return factory.apply(v);
            }
        }
        catch(Exception e) {

            throw new IllegalArgumentException("Config value ["+path+"] may not be parsed",e);
        }
    }

    /**
     * Extract a mandatory object parsed from a string out of a structured configuration.
     * @param config a parsed, structured configuration.
     * @param factory A factory method for creating instances like
     *        <code>URI::create</code> or <code>OffsetDateTime::parse</code>.
     * @param path A path, which might be dot-separated.
     * @return The parsed value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static <T> T getMandatoryParsed(Map<String,?> config, Function<? super String,T> factory, String path) {

        T t = getParsed(config,factory,path);

        if (t == null) {
            throw new IllegalArgumentException("Config value ["+path+"] has not been specified.");
        }

        return t;
    }

    /**
     * Extract a mandatory object parsed from a string out of a structured configuration.
     * @param config a parsed, structured configuration.
     * @param factory A factory method for creating instances like
     *        <code>URI::create</code> or <code>OffsetDateTime::parse</code>.
     * @param path A path, which might be dot-separated.
     * @return The parsed value or the default value, if nothing is found under the given path.
     */
    public static <T> T getParsed(Map<String,?> config, Function<? super String,T> factory, String path, T defaultValue) {

        T t = getParsed(config,factory,path);

        if (t == null) {
            return defaultValue;
        }

        return t;
    }

    /**
     * Get a list of parsed objects from a string list from the given path,
     * which contains a YAML list.
     * @param config a parsed, structured configuration.
     * @param factory A factory method for creating instances like
     *        <code>URI::create</code> or <code>OffsetDateTime::parse</code>.
     * @param path A path, which might be dot-separated.
     * @return A list of parsed objects or <code>null</code> if nothing is
     *         found under the given path.
     */
    public static <T> List<T> getParsedList(Map<String,?> config, Function<? super String,T> factory, String path) {

        List<?> lst = getList(config,path);

        if (lst == null) {
            return null;
        }

        return lst.stream().map((x) -> factory.apply(extractString(x))).collect(Collectors.toList());
    }

    /**
     * Get a list of parsed objects from a string list from the given path,
     * which contains a YAML list.
     * @param config a parsed, structured configuration.
     * @param factory A factory method for creating instances like
     *        <code>URI::create</code> or <code>OffsetDateTime::parse</code>.
     * @param path A path, which might be dot-separated.
     * @return A list of parsed objects.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static <T> List<T> getMandatoryParsedList(Map<String,?> config, Function<? super String,T> factory, String path) {

        List<T> ret = getParsedList(config,factory,path);

        if (ret == null) {
            throw new IllegalArgumentException("Config value ["+path+"] has not been specified.");
        }

        return ret;
    }

    /**
     * Extract an Integer from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The Integer or <code>null</code> if nothing is found under
     *         the given path.
     */
    public static Integer getInteger(Map<String,?> config, String path) {

        Object sub = getSubItem(config,path);

        try {
            return extractInteger(sub);
        }
        catch(Exception e) {
            throw new IllegalArgumentException("Config value ["+path+"] is not an integer",e);
        }
    }

    /**
     * Extract a mandatory int from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The int value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static int getMandatoryInt(Map<String,?> config, String path) {

        Integer i = getInteger(config,path);

        if (i == null) {
            throw new IllegalArgumentException("Config value ["+path+"] has not been specified.");
        }

        return i.intValue();
    }

    /**
     * Extract a mandatory int from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @param defaultValue A default value, if the given path does not exist.
     * @return The int value.
     */
    public static int getInt(Map<String,?> config, String path, int defaultValue) {

        Integer i = getInteger(config,path);

        if (i == null) {
            return defaultValue;
        }

        return i.intValue();
    }

    /**
     * Get a list of integers from the given path, which contains a YAML list.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return A list of integers or <code>null</code> if nothing is found under
     *         the given path.
     */
    public static List<Integer> getIntegerList(Map<String,?> config, String path) {

        List<?> lst = getList(config,path);

        if (lst == null) {
            return null;
        }

        return lst.stream().map((x) -> extractInteger(x)).collect(Collectors.toList());
    }

    /**
     * Get a list of integers from the given path, which contains a YAML list.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return A list of integers.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static List<Integer> getMandatoryIntegerList(Map<String,?> config, String path) {

        List<Integer> ret = getIntegerList(config,path);

        if (ret == null) {
            throw new IllegalArgumentException("Config value ["+path+"] has not been specified.");
        }

        return ret;
    }

    /**
     * Extract a Long from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The Long or <code>null</code> if nothing is found under
     *         the given path.
     */
    public static Long getLong(Map<String,?> config, String path) {

        Object sub = getSubItem(config,path);

        try {
            return extractLong(sub);
        }
        catch(Exception e) {
            throw new IllegalArgumentException("long config value ["+path+"] is not a long integer",e);
        }
    }

    /**
     * Extract a mandatory long from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The long value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static long getMandatoryLong(Map<String,?> config, String path) {

        Long i = getLong(config,path);

        if (i == null) {
            throw new IllegalArgumentException("long config value ["+path+"] has not been specified.");
        }

        return i.longValue();
    }

    /**
     * Extract a mandatory long from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @param defaultValue A default value, if the given path does not exist.
     * @return The long value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static long getLong(Map<String,?> config, String path, long defaultValue) {

        Long i = getLong(config,path);

        if (i == null) {
            return defaultValue;
        }

        return i.longValue();
    }

    /**
     * Get a list of longs from the given path, which contains a YAML list.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return A list of longs or <code>null</code> if nothing is found under
     *         the given path.
     */
    public static List<Long> getLongList(Map<String,?> config, String path) {

        List<?> lst = getList(config,path);

        if (lst == null) {
            return null;
        }

        return lst.stream().map((x) -> extractLong(x)).collect(Collectors.toList());
    }

    /**
     * Get a list of longs from the given path, which contains a YAML list.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return A list of longs.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static List<Long> getMandatoryLongList(Map<String,?> config, String path) {

        List<Long> ret = getLongList(config,path);

        if (ret == null) {
            throw new IllegalArgumentException("Config value ["+path+"] has not been specified.");
        }

        return ret;
    }

    /**
     * Extract a Double from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The Double or <code>null</code> if nothing is found under
     *         the given path.
     */
    public static Double getDouble(Map<String,?> config, String path) {

        Object sub = getSubItem(config,path);

        try {
            return extractDouble(sub);
        }
        catch(Exception e) {
            throw new IllegalArgumentException("double config value ["+path+"] is not a floating point constant",e);
        }
    }

    /**
     * Extract a mandatory double from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The double value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static double getMandatoryDouble(Map<String,?> config, String path) {

        Double i = getDouble(config,path);

        if (i == null) {
            throw new IllegalArgumentException("double config value ["+path+"] has not been specified.");
        }

        return i.doubleValue();
    }

    /**
     * Extract a mandatory double from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @param defaultValue A default value, if the given path does not exist.
     * @return The double value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static double getDouble(Map<String,?> config, String path, double defaultValue) {

        Double i = getDouble(config,path);

        if (i == null) {
            return defaultValue;
        }

        return i.doubleValue();
    }

    /**
     * Get a list of doubles from the given path, which contains a YAML list.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return A list of doubles or <code>null</code> if nothing is found under
     *         the given path.
     */
    public static List<Double> getDoubleList(Map<String,?> config, String path) {

        List<?> lst = getList(config,path);

        if (lst == null) {
            return null;
        }

        return lst.stream().map((x) -> extractDouble(x)).collect(Collectors.toList());
    }

    /**
     * Get a list of doubles from the given path, which contains a YAML list.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return A list of doubles.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static List<Double> getMandatoryDoubleList(Map<String,?> config, String path) {

        List<Double> ret = getDoubleList(config,path);

        if (ret == null) {
            throw new IllegalArgumentException("Config value ["+path+"] has not been specified.");
        }

        return ret;
    }

    /**
     * Extract a Boolean from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The Booelan or <code>null</code> if nothing is found under
     *         the given path.
     */
    public static Boolean getBoolean(Map<String,?> config, String path) {

        Object sub = getSubItem(config,path);

        try {
            return extractBoolean(sub);
        }
        catch(Exception e) {
            throw new IllegalArgumentException("Config value ["+path+"] is not a boolean",e);
        }
    }

    /**
     * Extract a mandatory boolean from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The boolean value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static boolean getMandatoryBoolean(Map<String,?> config, String path) {

        Boolean b = getBoolean(config,path);

        if (b == null) {
            throw new IllegalArgumentException("boolean config value ["+path+"] has not been specified.");
        }

        return b.booleanValue();
    }

    /**
     * Extract a mandatory boolean from a parsed structured configuration.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @param defaultValue A default value, if the given path does not exist.
     * @return The boolean value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static boolean getBoolean(Map<String,?> config, String path, boolean defaultValue) {

        Boolean b = getBoolean(config,path);

        if (b == null) {
            return defaultValue;
        }

        return b.booleanValue();
    }

    /**
     * Get a list of booleans from the given path, which contains a YAML list.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return A list of booleans or <code>null</code> if nothing is found under
     *         the given path.
     */
    public static List<Boolean> getBooleanList(Map<String,?> config, String path) {

        List<?> lst = getList(config,path);

        if (lst == null) {
            return null;
        }

        return lst.stream().map((x) -> extractBoolean(x)).collect(Collectors.toList());
    }

    /**
     * Get a list of booleans from the given path, which contains a YAML list.
     * @param config a parsed, structured configuration.
     * @param path A path, which might be dot-separated.
     * @return A list of booleans.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public static List<Boolean> getMandatoryBooleanList(Map<String,?> config, String path) {

        List<Boolean> ret = getBooleanList(config,path);

        if (ret == null) {
            throw new IllegalArgumentException("Config value ["+path+"] has not been specified.");
        }

        return ret;
    }

}
