/***********************************************************
 *
 * Service API of the clazzes.org project
 * https://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.api;

import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

/**
 * A wrapper around a configuration map, which eases access to
 * individual values via the methods provided by {@link ConfigurationHelper}
 */
public class ConfigWrapper {

    private final String pid;
    private final Map<String,?> content;

    public ConfigWrapper(String pid, Map<String, ?> content) {
        this.pid = pid;
        this.content = content;
    }

    @Override
    public String toString() {
        return "ConfigWrapper[pid="+this.pid+"]";
    }

    /**
     * @return The backing map of raw configuration values.
     */
    public Map<String,?> getContent() {
        return this.content;
    }

    /**
     * @return The set of keys in this map.
     * @see Map#keySet()
     */
    public Set<String> keySet() {
        return this.content.keySet();
    }

    private String subPid(String key) {
        return this.pid + "/" + key.replace('.','/');
    }

    /**
     * Extract a subtree from a configuration.
     * @param path A path, which might be dot-separated.
     * @return The subtree or <code>null</code>, if no such subtree exists.
     */
    public ConfigWrapper getSubTree(String path) {

        Map<String,?> sub = ConfigurationHelper.getSubTree(this.content,path);

        if (sub == null) {
            return null;
        }

        return new ConfigWrapper(this.subPid(path),sub);
    }

    /**
     * Extract a mandatory subtree from a configuration.
     * @param path A path, which might be dot-separated.
     * @return The subtree.
     * @throws IllegalArgumentException if the subtree does not exist.
     */
    public ConfigWrapper getMandatorySubTree(String path) {

        Map<String,?> sub = ConfigurationHelper.getMandatorySubTree(this.content,path);

        return new ConfigWrapper(this.subPid(path),sub);
    }

    /**
     * Extract a string from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The string or <code>null</code> if nothing is found under
     *         the given path.
     */
    public String getString(String path) {
        return ConfigurationHelper.getString(this.content,path);
    }

    /**
     * Extract a mandatory string from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The string.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public String getMandatoryString(String path) {
        return ConfigurationHelper.getMandatoryString(this.content,path);
    }

    /**
     * Extract a mandatory string from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @param defaultValue A default value, if the given path does not exist or the value
     *             is blank.
     * @return The string.
     */
    public String getString(String path, String defaultValue) {
        return ConfigurationHelper.getString(this.content,path,defaultValue);
    }


        /**
     * Get a list of strings from the given path, which contains a YAML list.
     * @param path A path, which might be dot-separated.
     * @return A list of strings or <code>null</code> if nothing is found under
     *         the given path.
     */
    public List<String> getStringList(String path) {

        return ConfigurationHelper.getStringList(this.content,path);
    }

    /**
     * Get a list of strings from the given path, which contains a YAML list.
     * @param path A path, which might be dot-separated.
     * @return A list of strings.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public List<String> getMandatoryStringList(String path) {

        return ConfigurationHelper.getMandatoryStringList(this.content,path);
    }

    /**
     * Extract an enum value from a parsed structured configuration.
     * @param cls The enum class to convert the value to.
     * @param path A path, which might be dot-separated.
     * @return The enum value or <code>null</code> if nothing is found under
     *         the given path.
     */
    public <T extends Enum<T>> T getEnum(Class<T> cls, String path) {
        return ConfigurationHelper.getEnum(this.content,cls,path);
    }

    /**
     * Extract a mandatory enum value from a parsed structured configuration.
     * @param cls The enum class to convert the value to.
     * @param path A path, which might be dot-separated.
     * @return The enum value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public <T extends Enum<T>> T getMandatoryEnum(Class<T> cls, String path) {
        return ConfigurationHelper.getMandatoryEnum(this.content,cls,path);
    }

    /**
     * Extract a mandatory enum value from a parsed structured configuration.
     * @param cls The enum class to convert the value to.
     * @param path A path, which might be dot-separated.
     * @param defaultValue The default value, if the given path is not defined.
     * @return The enum value.
     */
    public <T extends Enum<T>> T getEnum(Class<T> cls, String path, T defaultValue) {
        return ConfigurationHelper.getEnum(this.content,cls,path,defaultValue);
    }

    /**
     * Get a set of enum values from the given path, which contains a YAML list.
     * @param cls The enum class to convert the value to.
     * @param path A path, which might be dot-separated.
     * @return A set of enum values or <code>null</code> if nothing is found under
     *         the given path.
     */
    public <T extends Enum<T>> EnumSet<T> getEnumSet(Class<T> cls, String path) {

        return ConfigurationHelper.getEnumSet(this.content,cls,path);
    }

    /**
     * Get a set of enum values from the given path, which contains a YAML list.
     * @param cls The enum class to convert the value to.
     * @param path A path, which might be dot-separated.
     * @return A set of enum values.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public <T extends Enum<T>> EnumSet<T> getMandatoryEnumSet(Class<T> cls, String path) {

        return ConfigurationHelper.getMandatoryEnumSet(this.content,cls,path);
    }

    /**
     * Extract an object parsed from a string out of a structured configuration.
     * @param factory A factory method for creating instances like
     *        <code>URI::create</code> or <code>OffsetDateTime::parse</code>.
     * @param path A path, which might be dot-separated.
     * @return The parsed value or <code>null</code> if nothing is found under
     *         the given path.
     */
    public <T> T getParsed(Function<? super String,T> factory, String path) {
        return ConfigurationHelper.getParsed(this.content,factory,path);
    }

    /**
     * Extract a mandatory object parsed from a string out of a structured configuration.
     * @param factory A factory method for creating instances like
     *        <code>URI::create</code> or <code>OffsetDateTime::parse</code>.
     * @param path A path, which might be dot-separated.
     * @return The parsed value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public <T> T getMandatoryParsed(Function<? super String,T> factory, String path) {
        return ConfigurationHelper.getMandatoryParsed(this.content,factory,path);
    }

    /**
     * Extract a mandatory object parsed from a string out of a structured configuration.
     * @param factory A factory method for creating instances like
     *        <code>URI::create</code> or <code>OffsetDateTime::parse</code>.
     * @param path A path, which might be dot-separated.
     * @return The parsed value or the default value, if nothing is found under the given path.
     */
    public <T> T getParsed(Function<? super String,T> factory, String path, T defaultValue) {
        return ConfigurationHelper.getParsed(this.content,factory,path,defaultValue);
    }


    /**
     * Get a list of parsed objects from a string list from the given path,
     * which contains a YAML list.
     * @param factory A factory method for creating instances like
     *        <code>URI::create</code> or <code>OffsetDateTime::parse</code>.
     * @param path A path, which might be dot-separated.
     * @return A list of parsed objects or <code>null</code> if nothing is
     *         found under the given path.
     */
    public <T> List<T> getParsedList(Function<? super String,T> factory, String path) {

        return ConfigurationHelper.getParsedList(this.content,factory,path);
    }

    /**
     * Get a list of parsed objects from a string list from the given path,
     * which contains a YAML list.
     * @param factory A factory method for creating instances like
     *        <code>URI::create</code> or <code>OffsetDateTime::parse</code>.
     * @param path A path, which might be dot-separated.
     * @return A list of parsed objects.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public <T> List<T> getMandatoryParsedList(Function<? super String,T> factory, String path) {

        return ConfigurationHelper.getMandatoryParsedList(this.content,factory,path);
    }

    /**
     * Extract an Integer from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The Integer or <code>null</code> if nothing is found under
     *         the given path.
     */
    public Integer getInteger(String path) {
        return ConfigurationHelper.getInteger(this.content,path);
    }

    /**
     * Extract a mandatory int from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The int value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public int getMandatoryInt(String path) {
        return ConfigurationHelper.getMandatoryInt(this.content,path);
    }

    /**
     * Extract a mandatory int from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @param defaultValue A default value, if the given path does not exist.
     * @return The int value.
     */
    public int getInt(String path, int defaultValue) {
        return ConfigurationHelper.getInt(this.content,path,defaultValue);
    }

    /**
     * Get a list of integers from the given path, which contains a YAML list.
     * @param path A path, which might be dot-separated.
     * @return A list of integers or <code>null</code> if nothing is found under
     *         the given path.
     */
    public List<Integer> getIntegerList(String path) {

        return ConfigurationHelper.getIntegerList(this.content,path);
    }

    /**
     * Get a list of integers from the given path, which contains a YAML list.
     * @param path A path, which might be dot-separated.
     * @return A list of integers.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public List<Integer> getMandatoryIntegerList(String path) {

        return ConfigurationHelper.getMandatoryIntegerList(this.content,path);
    }

    /**
     * Extract a Long from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The Long or <code>null</code> if nothing is found under
     *         the given path.
     */
    public Long getLong(String path) {
        return ConfigurationHelper.getLong(this.content,path);
    }

    /**
     * Extract a mandatory long from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The long value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public long getMandatoryLong(String path) {
        return ConfigurationHelper.getMandatoryLong(this.content,path);
    }

    /**
     * Extract a mandatory long from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @param defaultValue A default value, if the given path does not exist.
     * @return The long value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public long getLong(String path, long defaultValue) {
        return ConfigurationHelper.getLong(this.content,path,defaultValue);
    }

    /**
     * Get a list of longs from the given path, which contains a YAML list.
     * @param path A path, which might be dot-separated.
     * @return A list of longs or <code>null</code> if nothing is found under
     *         the given path.
     */
    public List<Long> getLongList(String path) {

        return ConfigurationHelper.getLongList(this.content,path);
    }

    /**
     * Get a list of longs from the given path, which contains a YAML list.
     * @param path A path, which might be dot-separated.
     * @return A list of longs.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public List<Long> getMandatoryLongList(String path) {

        return ConfigurationHelper.getMandatoryLongList(this.content,path);
    }

    /**
     * Extract a Double from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The Double or <code>null</code> if nothing is found under
     *         the given path.
     */
    public Double getDouble(String path) {
        return ConfigurationHelper.getDouble(this.content,path);
    }

    /**
     * Extract a mandatory double from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The double value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public double getMandatoryDouble(String path) {
        return ConfigurationHelper.getMandatoryDouble(this.content,path);
    }

    /**
     * Extract a mandatory double from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @param defaultValue A default value, if the given path does not exist.
     * @return The double value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public double getDouble(String path, double defaultValue) {
        return ConfigurationHelper.getDouble(this.content,path,defaultValue);
    }

    /**
     * Get a list of doubles from the given path, which contains a YAML list.
     * @param path A path, which might be dot-separated.
     * @return A list of doubles or <code>null</code> if nothing is found under
     *         the given path.
     */
    public List<Double> getDoubleList(String path) {

        return ConfigurationHelper.getDoubleList(this.content,path);
    }

    /**
     * Get a list of doubles from the given path, which contains a YAML list.
     * @param path A path, which might be dot-separated.
     * @return A list of doubles.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public List<Double> getMandatoryDoubleList(String path) {

        return ConfigurationHelper.getMandatoryDoubleList(this.content,path);
    }

    /**
     * Extract a Boolean from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The Booelan or <code>null</code> if nothing is found under
     *         the given path.
     */
   public Boolean getBoolean(String path) {
        return ConfigurationHelper.getBoolean(this.content,path);
    }

    /**
     * Extract a mandatory boolean from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @return The boolean value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public boolean getMandatoryBoolean(String path) {
        return ConfigurationHelper.getMandatoryBoolean(this.content,path);
    }

    /**
     * Extract a mandatory boolean from a parsed structured configuration.
     * @param path A path, which might be dot-separated.
     * @param defaultValue A default value, if the given path does not exist.
     * @return The boolean value.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public boolean getBoolean(String path, boolean defaultValue) {
        return ConfigurationHelper.getBoolean(this.content,path,defaultValue);
    }

    /**
     * Get a list of booleans from the given path, which contains a YAML list.
     * @param path A path, which might be dot-separated.
     * @return A list of booleans or <code>null</code> if nothing is found under
     *         the given path.
     */
    public List<Boolean> getBooleanList(String path) {

        return ConfigurationHelper.getBooleanList(this.content,path);
    }

    /**
     * Get a list of booleans from the given path, which contains a YAML list.
     * @param path A path, which might be dot-separated.
     * @return A list of booleans.
     * @throws IllegalArgumentException if nothing is found under the given path.
     */
    public List<Boolean> getMandatoryBooleanList(String path) {

        return ConfigurationHelper.getMandatoryBooleanList(this.content,path);
    }

    /**
     * @return The config PID or the possible <code>PID/path</code> combination,
     *         if this is a sub configuration.
     */
    public String getPid() {
        return this.pid;
    }

}
