/***********************************************************
 *
 * Service API of the clazzes.org project
 * https://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.api;

import java.util.List;

/**
 * An information on a loaded configuration.
 */
public class ConfigPidInfo extends HasThrowableInfos implements Comparable<ConfigPidInfo> {

    private static final long serialVersionUID = 5718455838434871014L;

    private final String pid;
    private final Integer nkeys;
    private final Integer nlisteners;
    private final String location;

    /**
     * Construct a PID info with location and no throwable information.
     *
     * @param pid The config PIDs.
     * @param nkeys The number of key on the root config map.
     * @param nlisteners The number of listenres for this PID:
     * @param location The file location of this PID.
     */
    public ConfigPidInfo(String pid, Integer nkeys, Integer nlisteners, String location) {
        this.pid = pid;
        this.nkeys = nkeys;
        this.nlisteners = nlisteners;
        this.location = location;
    }

    /**
     * Construct a PID info with no location and no throwable information.
     *
     * @param pid The config PIDs.
     * @param nkeys The number of key on the root config map.
     * @param nlisteners The number of listenres for this PID:
     */
    public ConfigPidInfo(String pid, Integer nkeys, Integer nlisteners) {
        this(pid,nkeys,nlisteners,null);
    }

    /**
     * Construct a PID info with location and throwable information.
     *
     * @param pid The config PIDs.
     * @param nkeys The number of key on the root config map.
     * @param nlisteners The number of listenres for this PID:
     * @param location The file location of this PID.
     * @param throwables The exception infos of errors caught by listeners
     *                   for this PID.
     */
    public ConfigPidInfo(String pid, Integer nkeys, Integer nlisteners,
                         String location,
                         List<ThrowableInfo> throwables) {
        super(throwables);
        this.pid = pid;
        this.nkeys = nkeys;
        this.nlisteners = nlisteners;
        this.location = location;
    }

    /**
     * @return The PID of the configuration set.
     */
    public String getPid() {
        return this.pid;
    }

    /**
     * @return The number of config keys or <code>null</code>, if this
     *         PID is not yet loaded.
     */
    public Integer getNkeys() {
        return this.nkeys;
    }

    /**
     * @return The number of registered listeners or <code>null</code>,
     *         if this PID is not listened for.
     */
    public Integer getNlisteners() {
        return this.nlisteners;
    }

    /**
     * @return The location from which this configuration has been loaded.
     */
    public String getLocation() {
        return this.location;
    }

    @Override
    public int compareTo(ConfigPidInfo o) {

        String po = o.getPid();

        if (this.pid == null) {
            return po == null ? 0 : -1;
        }
        else if (po == null) {
            return 1;
        }

        return this.pid.compareTo(po);
    }
}
