/***********************************************************
 *
 * Service API of the clazzes.org project
 * https://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.api;

import java.util.ArrayList;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.logging.Logger;

/**
 * <p>A base class for components registering autocloseable listeners
 * and services.</p>
 *
 * <p>All methods in this class are not synchronized, please call them from
 * synchronized contexts, if needed.</p>
 */
public class ComponentSupport {

    Logger log = Logger.getLogger(ComponentSupport.class.getName());

    private List<AutoCloseable> listeners;
    private List<ServiceKey<?>> services;

    /**
     * <p>Add a service to the given service registry and to the list of registered
     * services to be dismantled by {@link #removeAllServices(ServiceRegistry)}.</p>
     *
     * @param <T> The service type parameter.
     * @param serviceRegistry The service registry
     * @param key The key of the service to be added.
     * @param iface The interface of the service.
     * @param service The serviec implementation.
     */
    public <T> void addService(ServiceRegistry serviceRegistry,
                               String key, Class<T> iface, T service) {

        serviceRegistry.addService(key,iface,service);
        if (this.services == null) {
            this.services = new ArrayList<ServiceKey<?>>();
        }

        this.services.add(new ServiceKey<T>(key,iface));
    }

    /**
     * <p>Remove all services, which have previously been registered by
     * {@link #addService(ServiceRegistry,String,Class,Object)} </p>
     * @param serviceRegistry The service registry to remove services from.
     */
    public void removeAllServices(ServiceRegistry serviceRegistry) {

        if (this.services != null) {

            for (int i=this.services.size()-1;i>=0;--i) {

                ServiceKey<?> sk = this.services.get(i);

                serviceRegistry.removeService(sk.getKey(),sk.getIface());
            }

            this.services = null;
        }
    }

    /**
     * Add a listener for later closing by {@link #closeAllListeners(BiConsumer)}
     * @param closeable A closeable returned by a listener method.
     */
    public void addListener(AutoCloseable closeable) {

        if (this.listeners == null) {
            this.listeners = new ArrayList<AutoCloseable>();
        }

        this.listeners.add(closeable);
    }

    /**
     * Close all listeners registers by {@link #addListener(AutoCloseable)}.
     * @param suppressedHandler A handle to be called when a close exception is
     *            suppressed, usually issuing a warning log message.
     */
    public void closeAllListeners(BiConsumer<AutoCloseable,Exception> suppressedHandler) {

        if (this.listeners != null) {
            for (int i=this.listeners.size()-1;i>=0;--i) {

                AutoCloseable listener = this.listeners.get(i);

                try {
                    listener.close();
                } catch (Exception e) {
                    suppressedHandler.accept(listener,e);
                }
            }
            this.listeners = null;
        }
    }

    /**
     * Close all listeners registers by {@link #addListener(AutoCloseable)} and
     * issue a warning log message to <code>java.util.logging</code>.
     */
    public void closeAllListeners() {

        if (this.listeners != null) {
            for (int i=this.listeners.size()-1;i>=0;--i) {

                AutoCloseable listener = this.listeners.get(i);

                try {
                    listener.close();
                } catch (Exception e) {
                    log.warning("Error closing listener "+listener+":"+e);
                }
            }
            this.listeners = null;
        }
    }

}
