/***********************************************************
 *
 * Service API of the clazzes.org project
 * https://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.api;

import java.util.List;

/**
 * Service for introspection of modules/components to be used by
 * SSH console commands.
 */
public interface ComponentManager {

    /**
     * @return The list of composnent layers.
     */
    List<ComponentLayerInfo> listLayers();

    /**
     * @return The list of modules found in all layers.
     */
    List<ModuleInfo> listModules();

    /**
     * @param layer The layer label to list the modules of.
     * @return The list of modules found in the given layer.
     */
    List<ModuleInfo> listModules(String layer);

    /**
     * @return The list of all known components.
     */
    List<ComponentInfo> listComponents();

    /**
     * @param filter A component name or a string
     *           like <code>layer/component</code>,
     *           <code>layer/*</code>,
     *           <code>layer/component@version</code>
     *           or <code>component@version</code>.
     * @return The list of modules found in the given layer.
     */
    List<ComponentInfo> listComponents(String filter);

    /**
     * Start a component.
     * @param component The component name or a string
     *           like <code>layer/component</code>,
     *           <code>layer/component@version</code>
     *           or <code>component@version</code>.
     */
    public void startComponent(String component);

    /**
     * Stop a component.
     * @param component The component name or a string
     *           like <code>layer/component</code>,
     *           <code>layer/component@version</code>
     *           or <code>component@version</code>.
     */
    public void stopComponent(String component);

    /**
     * <p>Reload the given component layer from the file system.</p>
     *
     * <p>This involves:</p>
     * <ol>
     * <li>Stop of all components in the layer.</li>
     * <li>Drop reference to layer in the Component Layer map.</li>
     * <li>Replace the layer with an instance freshly loaded from the filesystem.</li>
     * <li>Start all components in the new layer.</li>
     * </ol>
     *
     * <p>If classes and/or services of the existing layer are referenced by components
     * loaded before this call, they will still keep a reference to the old layer.</p>
     *
     * <p>Therefore, this method is mostly useful for top-level application layers.</p>
     *
     * @param label The label of the layer to reload.
     * @param timeout Timeout in millseconds to wait for the stop of components
     *                in the old layer.
     */
    public void reloadLayer(String label, long timeout);

    /**
     *
     * @param throwable A throwable to associate to a component by its stack trace.
     *           The resolved component will
     *           be transitioned to {@link ComponentState#CONFIG_LISTENER_FAILED}.
     */
    public void recordConfigException(ThrowableInfo throwable);

    /**
     *
     * @param throwable A throwable to associate to a component by its stack trace.
     *           The resolved component will
     *           be transitioned to {@link ComponentState#SERVICE_LISTENER_FAILED}.
     */
    public void recordServiceException(ThrowableInfo throwable);

    /**
     * Commit the current component resolved by the call stack.
     */
    public void commit();
}
