/***********************************************************
 *
 * Service API of the clazzes.org project
 * https://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.api;

import java.util.List;

/**
 * An information on a deployed java component.
 */
public class ComponentInfo extends HasThrowableInfos implements Comparable<ComponentInfo> {

    private static final long serialVersionUID = 4292571560841637366L;

    private final String name;
    private final int priority;
    private final ComponentState state;
    private final ModuleInfo module;

    /**
     * Construct a component info without throwable information.
     *
     * @param name The name of the component, which is the class name or
     *             a proxy description.
     * @param priority The servie priority.
     * @param state The current state of the component.
     * @param module The module of the compnent.
     */
    public ComponentInfo(String name, int priority, ComponentState state, ModuleInfo module) {
        this.name = name;
        this.priority = priority;
        this.state = state;
        this.module = module;
    }

    /**
     * Construct a component info with throwable information.
     *
     * @param name The name of the component, which is the class name or
     *             a proxy description.
     * @param priority The servie priority.
     * @param state The current state of the component.
     * @param module The module of the compnent.
     * @param throwables The trowable infos of exceptions cught by start/stop
     *                   of the component.
     */
    public ComponentInfo(String name, int priority, ComponentState state,
                         ModuleInfo module,
                         List<ThrowableInfo> throwables) {
        super(throwables);
        this.name = name;
        this.priority = priority;
        this.state = state;
        this.module = module;
    }

    public String getName() {
        return this.name;
    }

    public int getPriority() {
        return this.priority;
    }

    public ComponentState getState() {
        return this.state;
    }

    public ModuleInfo getModule() {
        return this.module;
    }

    @Override
    public int compareTo(ComponentInfo o) {

        if (this.getPriority() < o.getPriority()) {
            return -1;
        }
        else if (this.getPriority() > o.getPriority()) {
            return 1;
        }

        String no = o.getName();

        if (this.name == null) {
            return no == null ? 0 : -1;
        }
        else if (no == null) {
            return 1;
        }

        return this.name.compareTo(no);
    }
}
