/***********************************************************
 *
 * Login Components of
 *
 * https://www.clazzes.org
 *
 * Created: 04.04.2025
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.yubikey;

import org.clazzes.svc.api.Component;
import org.clazzes.svc.api.ComponentManager;
import org.clazzes.svc.api.ConfigurationEngine;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ServiceRegistry;
import org.clazzes.util.sec.TokenOtpChecker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class YubiKeyComponent implements Component {

    private static final Logger log = LoggerFactory.getLogger(YubiKeyComponent.class);

    public static final String CONFIG_PID = "org.clazzes.login.yubikey";
    public static final String SERVICE_KEY = "yubikey";

    private TokenOtpChecker tokenOtpChecker;

    protected void stopChecker(ServiceRegistry registry) {
        if (this.tokenOtpChecker != null) {
            registry.removeService(SERVICE_KEY,TokenOtpChecker.class);
            this.tokenOtpChecker = null;
        }

    }

    @Override
    public void start(ServiceContext context) throws Exception {

        ConfigurationEngine ce = context.getService(ConfigurationEngine.class).get();
        ServiceRegistry registry = context.getService(ServiceRegistry.class).get();
        ComponentManager componentManager = context.getService(ComponentManager.class).get();

        ce.listen(CONFIG_PID,config -> {

            YubiKeyOtpCheckerImpl oci = new YubiKeyOtpCheckerImpl();

            oci.setConnectTimeout(
                config.getInt("connectTimeoutMillis",60000));

            oci.setYubikeyVerifyLocation(
                config.getString("yubikeyVerifyLocation",
                                 "https://api.yubico.com/wsapi/2.0/verify"));

            oci.setYubikeyRequestorId(
                config.getString("yubikeyRequestorId",
                                 ""));

            log.info("Starting [{}]",oci);

            synchronized (this) {
                this.stopChecker(registry);
                registry.addService(SERVICE_KEY,TokenOtpChecker.class,oci);
                this.tokenOtpChecker = oci;
            }
        });

        // commit here, to be up even without a configured domain.
        componentManager.commit();
    }

    @Override
    public void stop(ServiceContext context) throws Exception {
        ServiceRegistry registry = context.getService(ServiceRegistry.class).get();

        synchronized (this) {
            this.stopChecker(registry);
        }
    }

}
