/**
 * $Id: YubiKeyOtpCheckerImpl.java 401 2017-10-10 12:43:57Z msrvmgr $
 *
 * (C) Copyright 2017 ITEG IT-Engineers GmbH
 */

package org.clazzes.login.yubikey;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.InterruptedIOException;
import java.io.LineNumberReader;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandlers;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.clazzes.util.http.UrlHelper;
import org.clazzes.util.sec.HashTools;
import org.clazzes.util.sec.TokenOtpChecker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * YubiKey check service implementation.
 */
public class YubiKeyOtpCheckerImpl implements TokenOtpChecker {

    private static final Logger log = LoggerFactory.getLogger(YubiKeyOtpCheckerImpl.class);

    private int connectTimeout = 60000;

    public void setConnectTimeout(int connectTimeout) {
        this.connectTimeout = connectTimeout;
    }

    private String yubikeyVerifyLocation;
    private String yubikeyRequestorId;


    @Override
	public boolean checkOTP(String otp, String[] knownIds) throws IOException {

        if (otp == null || otp.length() < 12) {
            return false;
        }

        String id = otp.substring(0,12);

        boolean otpOk = false;

        for (String knownId : knownIds) {

        	otpOk |= knownId.equals(id);
        }

        if (!otpOk) {

        	log.error("Yubikey ID [{}] is not in the list of known YubiKe IDs [{}].",id,Arrays.toString(knownIds));
            return false;
        }

        String nonce = HashTools.randomSalt(20,"0123456789abcdef");

        String url = UrlHelper.appendQueryParameterToUrl(this.yubikeyVerifyLocation,"otp",otp);
        url = UrlHelper.appendQueryParameterToUrl(url,"id",this.yubikeyRequestorId);
        url = UrlHelper.appendQueryParameterToUrl(url,"nonce",nonce);

        URI uri = URI.create(url);

        HttpRequest req =
            HttpRequest.newBuilder(uri).timeout(Duration.of(connectTimeout,ChronoUnit.MILLIS)).build();

        if (log.isDebugEnabled()) {
            log.debug("Performing Yubikey check request to [{}]...",url);
        }

        HttpClient client = HttpClient.newHttpClient();

        HttpResponse<InputStream> resp;
        try {
            resp = client.send(req,BodyHandlers.ofInputStream());
        } catch (InterruptedException e) {
            throw new InterruptedIOException("OTP Request to ["+url+"] has been interrupted.");
        }

        try (InputStream is = resp.body()) {

            int status =resp.statusCode();

            if (status == 200) {

                Map<String,String> kvs = new HashMap<String,String>();

                try (LineNumberReader reader = new LineNumberReader(new InputStreamReader(is,"UTF-8"))) {

                    String line;

                    while ((line=reader.readLine()) != null) {

                        if (line.isEmpty()) continue;

                        int epos = line.indexOf('=');

                        if (epos < 0) {
                            kvs.put(line,null);
                        }
                        else {
                            kvs.put(line.substring(0,epos),line.substring(epos+1));
                        }
                    }
                }

                if (log.isDebugEnabled()) {
                    log.debug("Yubikey check request to [{}] returned [{}]",url,kvs);
                }

                if (!"OK".equals(kvs.get("status"))) {
                    log.error("Yubikey check request to [{}] returned negative status [{}].",url,kvs.get("status"));
                }
                else if (!otp.equals(kvs.get("otp"))) {
                    log.error("Yubikey check request to [{}] returned non-matching OTP [{}].",url,kvs.get("otp"));
                }
                else if (!nonce.equals(kvs.get("nonce"))) {
                    log.error("Yubikey check request to [{}] returned non-matching nonce [{}].",url,kvs.get("nonce"));
                }
                else {
                    log.info("Yubikey check request to [{}] succeeded.",url);
                    return true;
                }
            }
            else {
                log.error("Yubikey check request to [{}] failed with HTTP reponse code [{}].",url,status);
            }
        }

        return false;
    }

    public void setYubikeyVerifyLocation(String yubikeyVerifyLocation) {
        this.yubikeyVerifyLocation = yubikeyVerifyLocation;
    }

    public void setYubikeyRequestorId(String yubikeyRequestorId) {
        this.yubikeyRequestorId = yubikeyRequestorId;
    }

}
