/***********************************************************
 *
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 20.03.2024
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

 package org.clazzes.login.oauth.impl;

import java.net.PasswordAuthentication;
import java.net.URI;
import java.time.Clock;
import java.time.Instant;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.Supplier;

import org.clazzes.login.jbo.jwt.JWToken;
import org.clazzes.login.oauth.DomainManager;
import org.clazzes.login.oauth.OAuthHttpClient;
import org.clazzes.login.oauth.OAuthRobotCredentials;
import org.clazzes.login.oauth.OAuthTokenResponse;
import org.clazzes.login.oauth.RobotGrantType;
import org.clazzes.login.oauth.TokenValidator;
import org.clazzes.login.oauth.i18n.OAuthMessages;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

//
// Concurrency tests are to be performed by
//
// for i in 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16
// do
//   SSH_AUTH_SOCK= ssh -i id-osgi-ed25519 -p 8104 localhost 'callFunction "(&(login.domain=WISKI)(login.mechanism=org.clazzes.login.oauth)(login.credentials=org.clazzes.util.sec.OAuthCredentials))"'&
// done
//
//

/**
 * Provides the OSGi service for robot tokens like
 *
 * <pre>
 * service; java.util.function.Supplier with properties:
 *    login.credentials = org.clazzes.util.sec.OAuthCredentials
 *    login.domain = WISKI
 *    login.mechanism = org.clazzes.login.oauth
 * </pre>
 */
public class RobotCredentialsSupplier implements Supplier<OAuthRobotCredentials> {

    private static final Logger log = LoggerFactory.getLogger(RobotCredentialsSupplier.class);

    private final OAuthHttpClient oauthHttpClient;
    private final TokenValidator tokenValidator;
    private final DomainManager domainManager;
    private final Clock clock;

    // thread-protected state
    // currently known credentials
    private OAuthRobotCredentials credentials;
    // is there an ongoing query for credentials?
    private boolean queryRunning;


    public RobotCredentialsSupplier(
        OAuthHttpClient oauthHttpClient,
        TokenValidator tokenValidator,
        DomainManager domainManager
    ) {
        this(oauthHttpClient, tokenValidator, domainManager, Clock.systemDefaultZone());
    }

    public RobotCredentialsSupplier(
        OAuthHttpClient oauthHttpClient,
        TokenValidator tokenValidator,
        DomainManager domainManager,
        Clock clock
    ) {
        this.oauthHttpClient = oauthHttpClient;
        this.tokenValidator = tokenValidator;
        this.domainManager = domainManager;
        this.clock = clock;
    }

    private OAuthRobotCredentials getInternal() throws Exception {


        String scope = this.domainManager.getDomainConfig().getRobotScope();
        PasswordAuthentication clientCredentials = this.domainManager.getDomainConfig().getRobotCredentials();

        URI tokenUri = this.domainManager.getTokenUri();

        OAuthTokenResponse resp;
        RobotGrantType grantType = this.domainManager.getDomainConfig().getRobotGrantType();

        if (grantType == RobotGrantType.client_credentials) {
            resp = this.oauthHttpClient.requestRobotToken(
                tokenUri,
                scope,
                clientCredentials,
                this.domainManager.getDomainConfig().getOptions()
            );
        }
        else if (grantType == RobotGrantType.password) {

            PasswordAuthentication userCred = this.domainManager.getDomainConfig().getRobotUserCredentials();

            if (userCred == null) {
                throw new IllegalStateException("No userId given for robot flow ["+grantType+"] for domain ["+
                                                this.domainManager.getDomainConfig().getDomain()+"]");
            }

            resp = this.oauthHttpClient.requestPasswordFlowToken(
                tokenUri,
                scope,
                clientCredentials,
                userCred,
                this.domainManager.getDomainConfig().getOptions()
            );
        }
        else {
            throw new IllegalStateException("Unsupported robot flow ["+grantType+"] for domain ["+
                                            this.domainManager.getDomainConfig().getDomain()+"]");
        }

        Map<String,String> additionalAttributes = null;

        if (resp.getIdToken() != null) {

            JWToken token = this.tokenValidator.validateRobotToken(this.domainManager, resp,OAuthMessages.getMesssages(Locale.ENGLISH));

            Map<String, Object> additionalClaims = token.getClaimSet().getAdditionalClaims();

            additionalAttributes = new HashMap<String,String>();

            for (Entry<String,Object> e : additionalClaims.entrySet()) {

                additionalAttributes.put(e.getKey(),e.getValue().toString());
            }
        }

        long expiry = resp.getTimestamp();

        Long maxAge = resp.getExpiresIn();

        if (maxAge == null) {

            log.warn("Got no expiry for robot token for domain [{}], falling back to 1 minute timeout.",this.domainManager.getDomain());
            expiry += 60000L;
        }
        else {
            // allow for 10% overlap.
            expiry += maxAge;
            expiry -= maxAge/10L;
        }

        return new OAuthRobotCredentials(resp.getAccessToken(),additionalAttributes,expiry);
    }

    @Override
    public OAuthRobotCredentials get() {

        try {
            synchronized(this) {

                if (this.queryRunning) {

                    if (log.isDebugEnabled()) {
                        log.debug("Waiting a minute for OAuth robot token query for domain [{}] in other thread...",
                                  this.domainManager.getDomain());
                    }

                    while(true) {
                        this.wait(60000L);

                        if (this.queryRunning) {
                            log.warn("OAuth robot token query for domain [{}] still running in other thread, waiting another minute...",
                                     this.domainManager.getDomain());
                        }
                        else {
                            break;
                        }
                    }

                    if (this.credentials == null) {
                        if (log.isDebugEnabled()) {
                            log.debug("OAuth robot token query for domain [{}] failed in other thread.",
                                      this.domainManager.getDomain());
                        }

                        throw new IllegalStateException("Unable to issue a robot token for domain ["+
                            this.domainManager.getDomain()+"]");
                    }

                    if (log.isDebugEnabled()) {
                        log.debug("OAuth robot token query for domain [{}] successfully finished in other thread.",
                                  this.domainManager.getDomain());
                    }

                    return this.credentials;
                }

                long now = Instant.now(this.clock).toEpochMilli();

                if (this.credentials != null && this.credentials.getRefreshMillis() > now) {

                    if (log.isDebugEnabled()) {
                        log.debug("Returning already issued OAuth robot token [{}] for domain [{}].",
                                  this.credentials,this.domainManager.getDomain());
                    }

                    return this.credentials;
                }

                if (log.isDebugEnabled() && this.credentials != null) {

                    log.debug("OAuth robot token [{}] for domain [{}] is expired, refreshing it now...",
                        this.credentials,this.domainManager.getDomain());
                }

                this.queryRunning = true;
            }

            OAuthRobotCredentials cred = null;

            try {
                if (log.isDebugEnabled()) {
                    log.debug("Querying new OAuth robot token for domain [{}]...",
                              this.domainManager.getDomain());
                }

                cred = this.getInternal();

                if (log.isDebugEnabled()) {
                    if (log.isDebugEnabled()) {
                        log.debug("Successfully issued new OAuth robot token [{}] for domain [{}].",
                                  cred,this.domainManager.getDomain());
                    }
                }
            }
            finally {
                synchronized(this) {
                    this.credentials = cred;
                    this.queryRunning = false;
                    this.notifyAll();
                }
            }
            return cred;
        }
        catch(Exception e) {

            log.error("Error issuing robot token for domain ["+this.domainManager.getDomain()+"]",e);
            throw new IllegalStateException("Unable to issue a robot token for domain ["+
                this.domainManager.getDomain()+"]");
        }
    }

}
