/***********************************************************
 * $Id$
 *
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 02.06.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.oauth;

import java.io.IOException;
import java.io.InputStream;
import java.time.Clock;
import java.util.Arrays;

import org.clazzes.login.jbo.json.JsonHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;

/**
 * Parser for {@link OAuthTokenResponse} and {@link OAuthTokenErrorResponse}
 * objects.
 */
public class OAuthTokenResponseParser {

    private static final Logger log = LoggerFactory.getLogger(OAuthTokenResponseParser.class);

    private static final JsonFactory JSON_FACTORY = new JsonFactory();

    static private OAuthTokenErrorResponse parseErrorInternal(JsonParser reader, String name) throws IOException {

        String ctxt = "oauth-error";

        String error=null;
        String errorDescription=null;
        String errorUri=null;
        String timestamp=null;
        String traceId=null;
        String correlationId=null;
        long[] errorCodes=null;

        while (true) {

            if (name == null) {
                name = JsonHelper.nextName(reader,ctxt);
                if (name == null) {
                    break;
                }
            }

            if ("error".equals(name)) {
                error = JsonHelper.nextString(reader,ctxt);
            }
            else if ("error_description".equals(name)) {
                errorDescription = JsonHelper.nextString(reader,ctxt);
            }
            else if ("error_uri".equals(name)) {
                errorUri = JsonHelper.nextString(reader,ctxt);
            }
            else if ("timestamp".equals(name)) {
                timestamp = JsonHelper.nextString(reader,ctxt);
            }
            else if ("trace_id".equals(name)) {
                traceId = JsonHelper.nextString(reader,ctxt);
            }
            else if ("correlation_id".equals(name)) {
                correlationId = JsonHelper.nextString(reader,ctxt);
            }
            else if ("error_codes".equals(name)) {

                long[] tmp = new long[32];
                int ncodes = 0;

                JsonHelper.beginArray(reader,ctxt);

                while (reader.nextToken() == JsonToken.VALUE_NUMBER_INT) {

                    long code = reader.getLongValue();

                    if (ncodes >= tmp.length) {
                        log.warn("Ignoring additonal error code [{}] (only 32 error codes are accepted in OAuth token error responses).",code);
                    }
                    else {
                        tmp[ncodes] = code;
                    }

                    ++ncodes;
                }

                JsonHelper.endArray(reader, ctxt);

                errorCodes = Arrays.copyOf(tmp,ncodes);
            }
            else {
                log.warn("Ignoring unknown property [{}] in OAuth token response.",name);
                reader.nextToken();
                reader.skipChildren();
            }

            name = null;
        }

        JsonHelper.endObject(reader,ctxt);

        return new OAuthTokenErrorResponse(error,errorDescription,errorUri,timestamp,traceId,correlationId,errorCodes);
    }

    /**
     * Parse an OAuth token response.
     * @param is An input stream to read from.
     * @param redirectUri The original redirect URI to be stored for later refreshes of the response.
     * @param state The original state to be stored for later refreshes of the response.
     * @param refreshToken The original refresh token to be stored for later reuse, if the response from
     *                     the server did not contain a new refresh token.
     * @return The parsed token response.
     * @throws OAuthTokenErrorResponse
     */
    public static OAuthTokenResponse parseResponse(InputStream is,String redirectUri, String state, String refreshToken, Clock clock) throws IOException, OAuthTokenErrorResponse {

        String ctxt = "oauth-token-response";

        try (JsonParser reader = JSON_FACTORY.createParser(is)) {

            String accessToken = null;
            String tokenType = null;
            Long expiresIn = null;
            String scope = null;
            String idToken = null;

            JsonHelper.beginObject(reader,ctxt);

            String name;

            while ((name = JsonHelper.nextName(reader,ctxt)) != null) {

                if ("error".equals(name) || "error_description".equals(name)) {
                    throw parseErrorInternal(reader,name);
                }
                else if ("access_token".equals(name)) {
                    accessToken = JsonHelper.nextString(reader,ctxt);
                }
                else if ("token_type".equals(name)) {
                    tokenType = JsonHelper.nextString(reader,ctxt);
                }
                else if ("expires_in".equals(name)) {
                    expiresIn = JsonHelper.nextLong(reader,ctxt) * 1000L;
                }
                else if ("scope".equals(name)) {
                    scope = JsonHelper.nextString(reader,ctxt);
                }
                else if ("refresh_token".equals(name)) {
                    refreshToken = JsonHelper.nextString(reader,ctxt);
                }
                else if ("id_token".equals(name)) {
                    idToken = JsonHelper.nextString(reader,ctxt);
                }
                else {
                    log.warn("Ignoring unknown property [{}] in OAuth token response.",name);
                    reader.nextToken();
                    reader.skipChildren();
                }
            }

            JsonHelper.endObject(reader,ctxt);

            return new OAuthTokenResponse(accessToken,tokenType,expiresIn,scope,refreshToken,idToken,redirectUri,state, clock);
        }
    }

    /**
     * Parse an OAuth token error response.
     * @param is An input stream to read from.
     * @return The parsed error response.
     */
    public static OAuthTokenErrorResponse parseErrorResponse(InputStream is) throws IOException {

        String ctxt = "oauth-error-response";

        try (JsonParser reader = JSON_FACTORY.createParser(is)) {

            JsonHelper.beginObject(reader,ctxt);

            return parseErrorInternal(reader,null);
        }
    }

}
