/***********************************************************
 * $Id$
 *
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 01.06.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.oauth;

import java.time.Clock;
import java.time.Instant;

/**
 * <p>
 * An OAuth token response.
 * </p>
 * <pre>
 * {
 *   "access_token": "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiIsIng1dCI6Ik5HVEZ2ZEstZnl0aEV1Q...",
 *   "token_type": "Bearer",
 *   "expires_in": 3599,
 *   "scope": "https%3A%2F%2Fgraph.microsoft.com%2Fmail.read",
 *   "refresh_token": "AwABAAAAvPM1KaPlrEqdFSBzjqfTGAMxZGUTdM0t4B4...",
 *   "id_token": "eyJ0eXAiOiJKV1QiLCJhbGciOiJub25lIn0.eyJhdWQiOiIyZDRkMTFhMi1mODE0LTQ2YTctOD...",
 * }
 * </pre>
 */
public class OAuthTokenResponse {

    private final String accessToken;
    private final String tokenType;
    private final Long expiresIn;
    private final String scope;
    private final String refreshToken;
    private final String idToken;
    private final long timestamp;
    private final String redirectUri;
    private final String state;

    public OAuthTokenResponse(String accessToken, String tokenType,
            Long expiresIn, String scope, String refreshToken, String idToken,
            String redirectUri, String state, Clock clock) {
        super();
        this.accessToken = accessToken;
        this.tokenType = tokenType;
        this.expiresIn = expiresIn;
        this.scope = scope;
        this.refreshToken = refreshToken;
        this.idToken = idToken;
        this.redirectUri = redirectUri;
        this.state = state;
        this.timestamp = Instant.now(clock).toEpochMilli();
    }

    /**
     * @return The access used for bearer authentication.
     */
    public String getAccessToken() {
        return this.accessToken;
    }

    /**
     * @return The token type, usually <code>Bearer</code>. <code>jwt</code> is used
     *          for tokens received from <code>Authorization: Bearer</code> headers from third
     *          parties with a known JWT structure.
     */
    public String getTokenType() {
        return this.tokenType;
    }

    /**
     * @return The optional expiry interval in milliseconds.
     */
    public Long getExpiresIn() {
        return this.expiresIn;
    }

    /**
     * @return The token's scope.
     */
    public String getScope() {
        return this.scope;
    }

    /**
     * @return The refresh token.
     */
    public String getRefreshToken() {
        return this.refreshToken;
    }

    /**
     * @return The JWT ID token, if available.
     */
    public String getIdToken() {
        return this.idToken;
    }

    /**
     * @return The millis-since-the-epoch timestamp of the time, when
     *         the response has been received.
     */
    public long getTimestamp() {
        return this.timestamp;
    }

    /**
     * @return The redirect URI specified at the token request.
     */
    public String getRedirectUri() {
        return this.redirectUri;
    }

    /**
     * @return The state specified at the token request.
     */
    public String getState() {
        return this.state;
    }

    @Override
    public String toString() {
        return "OAuthTokenResponse [accessToken=" + this.accessToken + ", tokenType="
                + this.tokenType + ", expiresIn=" + this.expiresIn + ", scope=" + this.scope
                + ", refreshToken=" + this.refreshToken + ", idToken=" + this.idToken
                + "]";
    }

}
