/***********************************************************
 * $Id$
 * 
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 01.06.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.oauth;

import java.util.Arrays;

import org.clazzes.util.aop.i18n.Messages;

/**
 * <p>
 * An OAuth token error response.
 * </p>
 * <pre>
 * {
 *   "error": "invalid_scope",
 *   "error_description": "AADSTS70011: The provided value for the input parameter 'scope' is not valid. The scope https://foo.microsoft.com/mail.read is not valid.\r\nTrace ID: 255d1aef-8c98-452f-ac51-23d051240864\r\nCorrelation ID: fb3d2015-bc17-4bb9-bb85-30c5cf1aaaa7\r\nTimestamp: 2016-01-09 02:02:12Z",
 *   "error_codes": [
 *     70011
 *   ],
 *   "timestamp": "2016-01-09 02:02:12Z",
 *   "trace_id": "255d1aef-8c98-452f-ac51-23d051240864",
 *   "correlation_id": "fb3d2015-bc17-4bb9-bb85-30c5cf1aaaa7"
 * }
 * </pre>
 */
public class OAuthTokenErrorResponse extends Exception {

    private static final long serialVersionUID = 1361636223592812770L;

    private final String errorDescription;
    private final String errorUri;
    private final String timestamp;
    private final String traceId;
    private final String correlationId;
    private final long[] errorCodes;
    
    public OAuthTokenErrorResponse(String error, Messages i18n) {
        this(error,i18n.getString(error),null,null,null,null,null);
    }

    public OAuthTokenErrorResponse(String error, String errorDescription) {
        this(error,errorDescription,null,null,null,null,null);
    }

    protected OAuthTokenErrorResponse(String error, String errorDescription,
            String errorUri,
            String timestamp, String traceId,
            String correlationId, long[] errorCodes) {
        super(error);
        this.errorDescription = errorDescription;
        this.errorUri = errorUri; 
        this.timestamp = timestamp;
        this.traceId = traceId;
        this.correlationId = correlationId;
        this.errorCodes = errorCodes;
    }
    
    /**
     * @return The error message, an alias for {@link #getMessage()}.
     */
    public String getError() {
        return this.getMessage();
    }

    /**
     * @return The error description.
     */
    public String getErrorDescription() {
        return this.errorDescription;
    }

    /**
     * @return The URI describing the error.
     */
    public String getErrorUri() {
        return this.errorUri;
    }

    /**
     * @return The timestamp of the response.
     */
    public String getTimestamp() {
        return this.timestamp;
    }

    /**
     * @return The trace ID.
     */
    public String getTraceId() {
        return this.traceId;
    }

    /**
     * @return The correlation ID.
     */
    public String getCorrelationId() {
        return this.correlationId;
    }

    /**
     * @return The list of numeric error codes, or <code>null</code> if not available.
     */
    public long[] getErrorCodes() {
        return this.errorCodes;
    }

    @Override
    public String toString() {
        return "OAuthTokenErrorResponse [error=" + this.getMessage()
                + "errorDescription=" + this.errorDescription
                + "errorUri=" + this.errorUri
                + ", timestamp=" + this.timestamp + ", traceId=" + this.traceId
                + ", correlationId=" + this.correlationId + ", errorCodes="
                + Arrays.toString(this.errorCodes) + "]";
    }
    
}
