/***********************************************************
 * $Id$
 *
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 04.06.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.oauth;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.clazzes.login.jbo.jwt.JWToken;
import org.clazzes.util.sec.DomainPrincipal;
import org.clazzes.util.sec.OAuthCredentials;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The actual principal, which originates from a successful OAuth
 * authentication.
 */
public class OAuthPrincipal implements DomainPrincipal, OAuthCredentials {

    private static final Logger log = LoggerFactory.getLogger(OAuthPrincipal.class);

    private static final String GROUPS_CLAIM = "groups";
    private static final String RESOURCE_ACCESS_CLAIM = "resource_access";
    private static final String RESOURCE_ROLES = "roles";

    private final String domain;
    private final String userName;
    private final String prettyName;
    private final String email;
    private final JWToken idToken;
    private final Map<String,String> additionalAttributes;
    private final String bearerToken;
    private final List<OAuthGroup> groups;

    private static String getAttribute(String fallback, Map<String,Object> attributes, String... names) {

        for (String name : names) {

            Object ret = attributes.get(name);

            if (ret != null) {
                return ret.toString();
            }
        }

        return fallback;
    }

    private static Map<String,String> getAdditionalAttribute(Map<String,Object> attributes) {

        Map<String,String> additionalAttributes = new HashMap<String,String>();

        for (Entry<String,Object> e : attributes.entrySet()) {

            if (!GROUPS_CLAIM.equals(e.getKey()) && e.getValue() != null) {
                additionalAttributes.put(e.getKey(),e.getValue().toString());
            }
        }

        return additionalAttributes;
    }

    private static List<OAuthGroup> getGroups(String domain, Map<String,Object> attributes, String groupRoleResource) {

        if (groupRoleResource == null) {

            Object groups_o = attributes.get(GROUPS_CLAIM);

            if (groups_o instanceof List<?>) {

                @SuppressWarnings("unchecked")
                List<String> groups = (List<String>)groups_o;

                List<OAuthGroup> ret = new ArrayList<OAuthGroup>(groups.size());

                for (String group:groups) {
                    ret.add(new OAuthGroup(domain,group));
                }

                return ret;
            }
            // fix for ADFS returning string instead of array of size 1, see #16 for details.
            else if (groups_o instanceof String) {
                return Collections.singletonList(new OAuthGroup(domain,groups_o.toString()));
            }
            else {

                return null;
            }
        }
        else {
            Object ra = attributes.get(RESOURCE_ACCESS_CLAIM);

            if (log.isDebugEnabled()) {
                log.debug("ra=[{}]",ra);
            }

            if (ra instanceof Map ra_map) {

                Object ac = ra_map.get(groupRoleResource);

                if (log.isDebugEnabled()) {
                    log.debug("ac=[{}]",ac);
                }

                if (ac instanceof Map ac_map) {
                    Object roles_o = ac_map.get(RESOURCE_ROLES);

                    if (log.isDebugEnabled()) {
                        log.debug("roles_o=[{}]",roles_o);
                    }

                    if (roles_o instanceof List){
                        @SuppressWarnings("unchecked")
                        List<String> roles = (List<String>)roles_o;

                        List<OAuthGroup> ret = new ArrayList<OAuthGroup>(roles.size());

                        for (String group:roles) {
                            ret.add(new OAuthGroup(domain,group));
                        }

                        return ret;
                    }
                }
            }
            return null;
        }
    }

    public OAuthPrincipal(String domain, Map<String,Object> attributes, String bearerToken, String groupRoleResource) {

        this.domain = domain;
        this.additionalAttributes = getAdditionalAttribute(attributes);

        this.groups = getGroups(domain,attributes,groupRoleResource);

        this.userName = getAttribute(null,attributes,"preferred_username","login","upn","email");
        this.email = getAttribute(null,attributes,"email");
        this.prettyName = getAttribute(null,attributes,"name","preferred_username","login","email","upn");

        this.bearerToken = bearerToken;
        this.idToken = null;
    }

    public OAuthPrincipal(String domain, JWToken idToken, String bearerToken, String groupRoleResource) {

        this.domain = domain;
        Map<String,Object> attributes = idToken.getClaimSet().getAdditionalClaims();
        this.additionalAttributes = getAdditionalAttribute(attributes);

        this.groups = getGroups(domain,attributes,groupRoleResource);

        this.userName = getAttribute(idToken.getClaimSet().getSubject(),attributes,"preferred_username","login","upn","email");
        this.email = getAttribute(null,attributes,"email");
        this.prettyName = getAttribute(null,attributes,"name","preferred_username","login","email","upn");

        this.bearerToken = bearerToken;
        this.idToken = idToken;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sec.DomainPrincipal#getName()
     */
    @Override
    public String getName() {

        return this.domain + "/" + this.userName;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sec.DomainPrincipal#getUserName()
     */
    @Override
    public String getUserName() {

        return this.userName;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sec.DomainPrincipal#getPrettyName()
     */
    @Override
    public String getPrettyName() {

        return this.prettyName;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sec.DomainPrincipal#getDomain()
     */
    @Override
    public String getDomain() {

        return this.domain;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sec.DomainPrincipal#getEMailAddress()
     */
    @Override
    public String getEMailAddress() {

        return this.email;
    }

    @Override
    public String getBearerToken() {
        return this.bearerToken;
    }

    public JWToken getIdToken() {
        return this.idToken;
    }

    public List<OAuthGroup> getGroups() {
        return this.groups;
    }

    @Override
    public Map<String,String> getAdditionalAttributes() {
        return this.additionalAttributes;
    }

    @Override
    public String getAdditionalAttribute(String key) {

        if (this.additionalAttributes == null) {
            return null;
        }

        return this.additionalAttributes.get(key);
    }

    @Override
    public String toString() {
        return "OAuthPrincipal [domain=" + this.domain + ", userName=" + this.userName +
                ", groups=" + this.groups +
                ", additionalAttributes=" + this.additionalAttributes + "]";
    }

}
