/***********************************************************
 * $Id$
 *
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 19.09.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/
package org.clazzes.login.oauth;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.Principal;
import java.text.ParseException;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import org.clazzes.login.oauth.i18n.OAuthMessages;
import org.clazzes.util.aop.i18n.Messages;
import org.clazzes.util.http.LocaleHelper;
import org.clazzes.util.http.RequestHelper;
import org.clazzes.util.http.UrlHelper;
import org.clazzes.util.http.sec.HttpLoginService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>The login servlet, which actually provides for the login form according
 * to the documentation for {@link HttpLoginService#getLoginUrl()}.</p>
 *
 * <p>When accepting a GET request, the login form is simply rendered based
 * on the login status of the current HTTP session.</p>
 */
public class OAuthLoginServlet extends OAuthAbstrServlet {

    private static final long serialVersionUID = 6376913713678650071L;

    private static final Logger log = LoggerFactory.getLogger(OAuthLoginServlet.class);

    private String alternativeLabel;
    private String alternativeIconLocation;

    /**
     * Actually render the login form according to {@link HttpLoginService#getLoginUrl()}.
     * @param i18n The internationalization object.
     * @param resp The HTTP resonse to write to.
     * @param css The CSS stylesheet to include. If <code>null</code>, the built-in
     *             stylesheet <code>oauth-login.css</code> will be used.
     * @param groups The formatted list of groups.
     * @throws IOException Upon I/O errors.
     * @throws ServletException Upon errors related to the servlet API.
     */
    protected void writeLoginForm(Messages i18n,
            HttpServletResponse resp,
            String css,
            List<String> domainsToSelect,
            String state,
            String selectedDomain,
            String user,
            String groups,
            OAuthTokenResponse tokens, OAuthTokenErrorResponse error) throws  IOException, ServletException {

        try {

            int status;

            if (user != null) {
                status = HttpServletResponse.SC_OK;
            }
            else if (error != null) {
                status = HttpServletResponse.SC_FORBIDDEN;
            }
            else {
                status = HttpServletResponse.SC_UNAUTHORIZED;
            }

            String lang = LocaleHelper.toXsLanguage(i18n.getLocale());

            resp.setHeader("X-Frame-Options","SAMEORIGIN");
            resp.setHeader("Content-Language",lang);
            resp.setHeader("Cache-Control","no-cache");
            resp.setHeader("Pragma","no-cache");
            resp.setHeader("Expires","0");
            resp.setContentType("application/xhtml+xml");

            resp.getOutputStream().write("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" \"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n".getBytes("UTF-8"));

            XMLStreamWriter xsw = xmlOutputFactory.createXMLStreamWriter(resp.getOutputStream(),"UTF-8");

            xsw.setDefaultNamespace(XHTML_NS_URI);

            //xsw.writeStartDocument();

            xsw.writeStartElement("html");
            xsw.writeDefaultNamespace(XHTML_NS_URI);
            xsw.writeAttribute("lang",lang);
            xsw.writeAttribute("xml:lang",lang);
            xsw.writeStartElement("head");

            xsw.writeEmptyElement("meta");
            xsw.writeAttribute("http-equiv","Content-Type");
            xsw.writeAttribute("content","application/xhtml+xml");

            xsw.writeEmptyElement("link");
            xsw.writeAttribute("type","text/css");
            xsw.writeAttribute("rel","stylesheet");
            xsw.writeAttribute("href",css == null ? "oauth-login.css" : css);

            if (domainsToSelect == null && status == HttpServletResponse.SC_UNAUTHORIZED) {

                xsw.writeStartElement("script");

                StringBuffer script = new StringBuffer();

                script.append("\nsetTimeout(function() {\n");
                script.append("  var loc = window.location.href;\n");
                script.append("  var r = loc.match(/&n=([0-9]+)/);\n");
                script.append("  if (r == null) {\n");
                script.append("    loc += '&n=1';\n");
                script.append("  }\n");
                script.append("  else {\n");
                script.append("    var n = Number(r[1])+1;\n");
                script.append("    loc = loc.replace(/&n=[0-9]+/,'&n='+n);\n");
                script.append("  }\n");
                script.append("  window.location.href = loc;\n");
                script.append("},100);\n");

                xsw.writeCData(script.toString());
                xsw.writeEndElement();
            }

            if (domainsToSelect != null) {

                xsw.writeStartElement("script");

                StringBuffer script = new StringBuffer();

                script.append("\nfunction openDomain(domain) {\n");
                script.append("  window.open('/oauth-login/start?locale=");
                script.append(lang);
                script.append("&domain='+domain+'&state=");
                script.append(state);
                script.append("');\n");
                script.append("  var loc = window.location.href;\n");
                script.append("  loc += loc.indexOf('?')>=0 ? '&' : '?';\n");
                script.append("  loc += 'state=';\n");
                script.append("  loc += '");
                script.append(state);
                script.append("';\n");
                script.append("  window.location.href = loc;\n");
                script.append("}\n");

                xsw.writeCData(script.toString());

                xsw.writeEndElement();
            }

            if (status == HttpServletResponse.SC_OK &&
                tokens != null &&
                tokens.getAccessToken() != null) {

                xsw.writeStartElement("script");

                StringBuffer script = new StringBuffer();

                script.append("\nasync function writeAccessToken(pfx,usage) {\n");
                script.append("  try {\n");
                script.append("    await navigator.clipboard.writeText(pfx+'\"");
                script.append(tokens.getAccessToken());
                script.append("\"');\n");
                script.append("    document.getElementById('http-login-access-feedback').innerText='Token copied to clipboard.';");
                script.append("    document.getElementById('http-login-access-usage').innerText=usage;");
                script.append("  } catch (error) {\n");
                script.append("    console.log('Copying token failed',error);\n");
                script.append("    document.getElementById('http-login-access-feedback').innerText='Copying token failed: '+error.messages;\n");
                script.append("    document.getElementById('http-login-access-usage').innerText='';");
                script.append("  }\n");

                script.append("}\n");

                xsw.writeCData(script.toString());

                xsw.writeEndElement();
            }

            xsw.writeStartElement("title");
            xsw.writeCharacters("OAuth Single-Sign-On");
            xsw.writeEndElement(); // </title>

            xsw.writeEndElement(); // </head>

            xsw.writeStartElement("body");

            // hidden form
            xsw.writeStartElement("form");
            xsw.writeAttribute("id","loginResultForm");

            xsw.writeEmptyElement("input");
            xsw.writeAttribute("type","hidden");
            xsw.writeAttribute("name","status");
            xsw.writeAttribute("value",String.valueOf(status));

            xsw.writeEmptyElement("input");
            xsw.writeAttribute("type","hidden");
            xsw.writeAttribute("name","principal");
            xsw.writeAttribute("value",user == null ? "" : user);

            xsw.writeEndElement(); // </form>

            // login form

            if (status == HttpServletResponse.SC_OK) {

                xsw.writeStartElement("p");

                if (groups == null) {
                    xsw.writeCharacters(i18n.formatString("logged-in-as-ok",user));
                }
                else {
                    xsw.writeCharacters(i18n.formatString("logged-in-as-ok-with-groups",user,groups));
                }
                xsw.writeEndElement(); // </p>

                if (tokens != null) {
                    if (tokens.getAccessToken() != null) {
                        xsw.writeStartElement("p");

                        xsw.writeCharacters("Copy access token: ");

                        xsw.writeStartElement("a");
                        xsw.writeAttribute("href","javascript:writeAccessToken('TOKEN=','curl -H \"Authorization: Bearer \" -H \"Accept: application/json\" https://host/path')");
                        xsw.writeAttribute("class","http-login-action");
                        xsw.writeCharacters("unix");
                        xsw.writeEndElement(); // </a>

                        xsw.writeCharacters(" ");

                        xsw.writeStartElement("a");
                        xsw.writeAttribute("href","javascript:writeAccessToken(' = ','Invoke-WebRequest -Headers  @{ Authorization=\"Bearer \"; Accept=\"application/json\" } -Uri https://host/path')");
                        xsw.writeAttribute("class","http-login-action");
                        xsw.writeCharacters("ps1");
                        xsw.writeEndElement(); // </a>

                        xsw.writeEndElement(); // </p>

                        xsw.writeStartElement("p");
                        xsw.writeAttribute("id","http-login-access-feedback");
                        xsw.writeEndElement(); // </p>

                        xsw.writeStartElement("p");
                        xsw.writeAttribute("id","http-login-access-usage");
                        xsw.writeEndElement(); // </p>
                    }
                }

                xsw.writeStartElement("p");
                xsw.writeStartElement("a");
                xsw.writeAttribute("href",this.oauthHttpLoginService.getLoginUrl() + "?logout=true&locale=" + lang);
                xsw.writeAttribute("class","http-login-action");
                xsw.writeCharacters(i18n.getString("do-logout"));
                xsw.writeEndElement(); // </a>
                xsw.writeEndElement(); // </p>
            }
            else if (domainsToSelect != null) {

                xsw.writeStartElement("div");
                xsw.writeAttribute("class","http-login-Domain");
                xsw.writeCharacters(i18n.getString("select-domain"));
                xsw.writeEndElement(); // </div>

                for (String domain : domainsToSelect) {

                    DomainManager dm = this.configurationService.getDomainManager(domain);
                    DomainConfig dc = dm.getDomainConfig();

                    URI favicon = dc.getFaviconLocation();

                    xsw.writeStartElement("div");
                    xsw.writeAttribute("class","http-login-Domain");

                    if (favicon != null) {

                        xsw.writeStartElement("img");
                        xsw.writeAttribute("width","16");
                        xsw.writeAttribute("height","16");
                        xsw.writeAttribute("src",favicon.toString());
                        xsw.writeEndElement();
                    }

                    xsw.writeStartElement("a");
                    xsw.writeAttribute("href","javascript:openDomain('"+domain+"')");
                    xsw.writeAttribute("class","http-login-action");
                    xsw.writeCharacters(dc.getLabel());
                    xsw.writeEndElement(); // </a>

                    if (dc.getOptions().contains(ConfigOptions.renderUserLogout)) {

                        try {
                            URI logoutUri = dm.getOpenIdLocation("end_session_endpoint");

                            xsw.writeStartElement("a");
                            xsw.writeAttribute("href",logoutUri.toString());
                            xsw.writeAttribute("target","_blank");
                            xsw.writeAttribute("class","http-login-Logout");
                            xsw.writeAttribute("title",i18n.formatString("logout-from-target-domain",domain));
                            xsw.writeEndElement(); // </a>

                        } catch (IllegalStateException|URISyntaxException e) {
                            log.warn("Unable to render logout URI for domain ["+domain+"]", e);
                        }
                    }

                    xsw.writeEmptyElement("br"); // <br/>

                    xsw.writeEndElement(); // </div>
                }

                HttpLoginService alternativeService = this.oauthHttpLoginService.getAlternativeService();

                if (alternativeService != null) {

                    String href = alternativeService.getLoginUrl();

                    href = UrlHelper.appendQueryParameterToUrl(href,"oauth","true");
                    href = UrlHelper.appendQueryParameterToUrl(href,"locale",lang);

                    xsw.writeStartElement("div");
                    xsw.writeAttribute("class","http-login-Domain");

                    if (this.alternativeIconLocation != null && !this.alternativeIconLocation.isBlank()) {

                        xsw.writeStartElement("img");
                        xsw.writeAttribute("width","16");
                        xsw.writeAttribute("height","16");
                        xsw.writeAttribute("src",this.alternativeIconLocation);
                        xsw.writeEndElement();
                    }

                    xsw.writeStartElement("a");
                    xsw.writeAttribute("href",href);
                    xsw.writeAttribute("class","http-login-action");

                    if (this.alternativeLabel != null && !this.alternativeLabel.isBlank()) {
                        xsw.writeCharacters(this.alternativeLabel);
                    }
                    else {
                        xsw.writeCharacters(i18n.getString("alternative-login"));
                    }

                    xsw.writeEndElement(); // </a>
                    xsw.writeEmptyElement("br"); // <br/>
                    xsw.writeEndElement(); // </div>
                }
            }
            else if (error != null) {
                xsw.writeCharacters(i18n.formatString("authentication-failed",selectedDomain));

                xsw.writeStartElement("p");
                xsw.writeCharacters(error.getError());

                if (error.getErrorDescription() != null) {
                    xsw.writeEmptyElement("br");
                    xsw.writeCharacters(error.getErrorDescription());
                }

                xsw.writeEndElement();

                xsw.writeStartElement("a");
                xsw.writeAttribute("href",this.oauthHttpLoginService.getLoginUrl());
                xsw.writeAttribute("class","http-login-action");
                xsw.writeCharacters(i18n.getString("do-retry"));
                xsw.writeEndElement(); // </a>
            }
            else {
                xsw.writeStartElement("p");
                xsw.writeCharacters(i18n.formatString("authentication-running",selectedDomain));
                xsw.writeEndElement(); // </p>

                xsw.writeStartElement("p");
                xsw.writeStartElement("a");
                xsw.writeAttribute("href",this.oauthHttpLoginService.getLoginUrl() + "?logout=true&locale=" + lang +"&state=" + state);
                xsw.writeAttribute("class","http-login-action");
                xsw.writeCharacters(i18n.getString("abort-authentication"));
                xsw.writeEndElement(); // </a>
                xsw.writeEndElement(); // </p>
            }

            xsw.writeEndElement(); // </body>

            xsw.writeEndElement(); // </html>

            xsw.writeEndDocument();
            xsw.close();

            resp.flushBuffer();

        } catch (XMLStreamException e) {

            throw new ServletException("Error setting XML stream writer",e);
        }

    }

    protected static String makeGroupString(OAuthPrincipal principal) {

        List<OAuthGroup> groups = principal.getGroups();

        String groupString = null;

        if (groups != null) {
            groupString =
                groups.stream().
                    map(x -> x.getName()).
                    collect(Collectors.joining(","));
        }

        if (log.isDebugEnabled()) {
            log.debug("groupString=[{}]",groupString);
        }

        return groupString;
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException {

        // state my be set by login iframe reload or successful response to oauth authorization request,
        String state = req.getParameter("state");
        String css = req.getParameter("css");
        Locale locale = getRequestLocale(req);
        Messages i18n = OAuthMessages.getMesssages(locale);
        String pi = req.getPathInfo();

        URI requestURI;

        try {
            requestURI = RequestHelper.getOriginalRequestUri(req);
        } catch (ParseException | URISyntaxException e) {
           throw new ServletException("Unable to parse full request URI for request to ["+RequestHelper.getRequestUrl(req)+"]",e);
        }

        if (log.isDebugEnabled()) {
            log.debug("Received request to [{}]",requestURI);
            log.debug("domains=",this.configurationService.getDomains());
        }

        if (pi == null) {

            String logout = req.getParameter("logout");

            if ("true".equals(logout)) {

                this.oauthHttpLoginService.logout(req);

                if (state != null) {
                    this.authStateCache.removeAuthState(state);
                }

                resp.sendRedirect(this.oauthHttpLoginService.getLoginUrl() + "?locale=" + LocaleHelper.toXsLanguage(locale));
                return;
            }

            LoginInfo loginInfo = this.oauthHttpLoginService.checkLoginInfo(req);

            if (loginInfo != null) {

                OAuthPrincipal principal = loginInfo.getPrincipal();

                String groupString = makeGroupString(principal);

                boolean show_tokens = Boolean.TRUE.equals(RequestHelper.getBooleanParameter(req,"tokens"));

                this.writeLoginForm(i18n,resp,css,null,null,principal.getDomain(),principal.getName(),
                                    groupString,show_tokens?loginInfo.getResponse():null, null);
                return;
            }

            AuthState authState = null;

            // if we have a state, check for result of ongoing authorization/authentication.
            if (state != null) {

                authState = this.authStateCache.getAuthState(state);

                if (authState == null) {
                    log.warn("Received request to [{}] with invalid or expired state parameter.",requestURI);
                    resp.sendRedirect(this.oauthHttpLoginService.getLoginUrl() + "?locale=" + LocaleHelper.toXsLanguage(locale));
                    return;
                }
            }

            if (authState == null) {

                HttpLoginService alternativeService = this.oauthHttpLoginService.getAlternativeService();

                if (alternativeService != null) {
                    Principal principal = alternativeService.checkLogin(req);
                    if (principal != null) {
                        resp.sendRedirect(alternativeService.getLoginUrl());
                        return;
                    }
                }

                // render domain selection, if no domain selected.
                List<String> domains = this.configurationService.getDomains();

                authState = this.authStateCache.createAuthState(locale,900000L);

                this.writeLoginForm(i18n,resp,css,domains,authState.getState(),null,null,null,null, null);
                return;
            }
            else {
                try {

                    Integer n = RequestHelper.getIntegerParameter(req,"n");

                    long timeout = n == null ? 0L : 30000L;

                    OAuthTokenResponse response = authState.waitForResponse(timeout);
                    String user = null;
                    String groupString = null;

                    if (response != null) {
                        log.info("Retrieving logged on principal for request to [{}].",requestURI);

                        OAuthPrincipal principal = this.oauthHttpLoginService.tryLogin(req,resp,authState.getDomain(),response,i18n);

                        user = principal.getName();

                        groupString = makeGroupString(principal);

                        log.info("Successful login of user [{}] upon request to [{}].",user,requestURI);

                        this.authStateCache.removeAuthState(state);
                    }

                    boolean show_tokens = Boolean.TRUE.equals(RequestHelper.getBooleanParameter(req,"tokens"));

                    this.writeLoginForm(i18n,resp,css,null,state,authState.getDomain(),
                                        user,groupString,show_tokens?response:null, null);

                    return;

                } catch (OAuthTokenErrorResponse e) {

                    this.writeLoginForm(i18n,resp,css,null,null,authState.getDomain(),null,null,null, e);
                    return;

                } catch (InterruptedException e) {

                    throw new ServletException("Wait for ongoing authorization has been interrupted",e);
                }
            }
        }
        else {
            resp.sendError(HttpServletResponse.SC_NOT_FOUND);
        }
    }

    @Override
    public String getServletInfo() {

        return OAuthLoginServlet.class.getSimpleName();
    }

    public void setAlternativeLabel(String alternativeLabel) {
        this.alternativeLabel = alternativeLabel;
    }

    public void setAlternativeIconLocation(String alternativeIconLocation) {
        this.alternativeIconLocation = alternativeIconLocation;
    }

}
