/***********************************************************
 *
 * Service API of the clazzes.org project
 * https://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.oauth;

import java.util.regex.Pattern;

import org.clazzes.login.oauth.impl.JWTokenValidator;
import org.clazzes.login.oauth.impl.OAuthHttpClientImpl;
import org.clazzes.svc.api.Component;
import org.clazzes.svc.api.ComponentManager;
import org.clazzes.svc.api.ComponentSupport;
import org.clazzes.svc.api.ConfigurationEngine;
import org.clazzes.svc.api.CoreService;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ServiceRegistry;
import org.clazzes.util.http.AdditionalHeader;
import org.clazzes.util.http.ResourceServlet;
import org.clazzes.util.http.sec.HttpLoginService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.servlet.Servlet;

public class OAuthLoginComponent extends ComponentSupport implements Component {

    private static final Logger log = LoggerFactory.getLogger(OAuthLoginComponent.class);

    public static final String LOGIN_MECHANISM = "org.clazzes.login.oauth";

    private static final int DEFAULT_SESSION_TIMEOUT = 180;
    private static final String DEFAULT_SESSION_COOKIE = "OAUTH_SESSION_ID";
    private static final boolean DEFAULT_SECURE_COOKIE = false;
    private static final SameSitePolicy DEFAULT_SAME_SITE_POLICY = SameSitePolicy.Strict;
    private static final TokenType DEFAULT_DELEGATE_TOKEN_TYPE = TokenType.OPAQUE;

    private GarbageCollector gc;

    @Override
    public void start(ServiceContext context) throws Exception {

        CoreService cs = context.getService(CoreService.class).get();
        ServiceRegistry serviceRegistry = context.getService(ServiceRegistry.class).get();
        ConfigurationEngine ce = context.getService(ConfigurationEngine.class).get();
        ComponentManager componentManager = context.getService(ComponentManager.class).get();

        OAuthHttpClientImpl httpClient = new OAuthHttpClientImpl();

        TokenGenerator tokenGenerator = new TokenGenerator();

        LoginInfoCache lic = new LoginInfoCache();
        lic.setTokenGenerator(tokenGenerator);

        AuthStateCache asc = new AuthStateCache();
        asc.setTokenGenerator(tokenGenerator);

        this.gc = new GarbageCollector();

        this.gc.setAuthStateCache(asc);
        this.gc.setLoginInfoCache(lic);

        TokenValidator tokenValidator = new JWTokenValidator();

        ConfigurationService configurationService = new ConfigurationService(serviceRegistry);

        configurationService.setOauthHttpClient(httpClient);
        configurationService.setTokenValidator(tokenValidator);

       this.addListener(ce.listen(ConfigurationService.CONFIG_PID,(config) -> { synchronized(this) {

            this.removeAllServices(serviceRegistry);

            configurationService.accept(config);

            this.gc.executorServiceBound(cs.getScheduledExecutorService());

            int sessionTimeout = config.getInt("sessionTimeout",DEFAULT_SESSION_TIMEOUT);
            String sessionCookie = config.getString("sessionCookie",DEFAULT_SESSION_COOKIE);
            boolean secureCookie = config.getBoolean("secureCookie", DEFAULT_SECURE_COOKIE);
            SameSitePolicy sameSitePolicy = config.getEnum(SameSitePolicy.class, "sameSitePolicy",DEFAULT_SAME_SITE_POLICY);

            String delegateDomain =  config.getString("delegateDomain");
            TokenType delegateTokenType =  config.getEnum(TokenType.class,"delegateTokenType",DEFAULT_DELEGATE_TOKEN_TYPE);

            String alternativeMechanism = config.getString("alternativeMechanism");
            String alternativeLabel = config.getString("alternativeLabel");
            String alternativeIconLocation = config.getString("alternativeIconLocation");

            OAuthHttpLoginService loginService = new OAuthHttpLoginService(serviceRegistry);

            loginService.setAlternativeMechanism(alternativeMechanism);
            loginService.setConfigurationService(configurationService);
            loginService.setDelegateDomain(delegateDomain);
            loginService.setDelegateTokenType(delegateTokenType);
            loginService.setLoginInfoCache(lic);
            loginService.setOauthHttpClient(httpClient);
            loginService.setSameSitePolicy(sameSitePolicy);
            loginService.setSecureCookie(secureCookie);
            loginService.setSessionCookie(sessionCookie);
            loginService.setSessionTimeout(sessionTimeout);
            loginService.setTokenValidator(tokenValidator);

            this.addService(serviceRegistry,LOGIN_MECHANISM,HttpLoginService.class,loginService);

            OAuthStartServlet startServlet = new OAuthStartServlet();
            startServlet.setAuthStateCache(asc);
            startServlet.setConfigurationService(configurationService);
            startServlet.setOauthHttpClient(httpClient);
            startServlet.setOauthHttpLoginService(loginService);
            this.addService(serviceRegistry,"/oauth-login/start",Servlet.class,startServlet);

            OAuthAuthServlet authServlet = new OAuthAuthServlet();
            authServlet.setAuthStateCache(asc);
            authServlet.setConfigurationService(configurationService);
            authServlet.setOauthHttpClient(httpClient);
            authServlet.setOauthHttpLoginService(loginService);
            this.addService(serviceRegistry,"/oauth-login/auth",Servlet.class,authServlet);

            OAuthLoginServlet loginServlet = new OAuthLoginServlet();
            loginServlet.setAlternativeIconLocation(alternativeIconLocation);
            loginServlet.setAlternativeLabel(alternativeLabel);
            loginServlet.setAuthStateCache(asc);
            loginServlet.setConfigurationService(configurationService);
            loginServlet.setOauthHttpClient(httpClient);
            loginServlet.setOauthHttpLoginService(loginService);
            this.addService(serviceRegistry,"/oauth-login/login",Servlet.class,loginServlet);


            ResourceServlet resourceServlet = new ResourceServlet();
            resourceServlet.setMaxAgeSeconds(3600);
            resourceServlet.addAdditionalHeader(
                new AdditionalHeader("X-Frame-Options",
                                     "SAMEORIGIN",
                                     null,
                                     Pattern.compile("text/html")));

            resourceServlet.setResourceModule(OAuthLoginServlet.class.getModule());
            resourceServlet.setResourcePath("/META-INF/webapp/oauth-login");

            resourceServlet.addExcludeMimeType("image/png");
            resourceServlet.addExcludeMimeType("image/gif");
            resourceServlet.addExcludeMimeType("image/jpeg");
            resourceServlet.addExcludeMimeType("image/tiff");

            this.addService(serviceRegistry,"/oauth-login/*",Servlet.class,resourceServlet);
        }}));

        // commit here, to be up even without a configured domain.
        componentManager.commit();
    }

    @Override
    public void stop(ServiceContext context) throws Exception {

        if (this.gc != null) {
            this.gc.executorServiceUnbound();
            this.gc = null;
        }

        this.closeAllListeners((c,e)->{
            log.warn("Error closing OAuth listener [{}]",c,e);
        });

        ServiceRegistry serviceRegistry = context.getService(ServiceRegistry.class).get();
        this.removeAllServices(serviceRegistry);
    }

}
