/***********************************************************
 * $Id$
 *
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 19.09.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.oauth;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TimeZone;

import org.clazzes.util.http.LocaleHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A cache for login infos.
 */
public class LoginInfoCache {

    private static final Logger log = LoggerFactory.getLogger(LoginInfoCache.class);

    private final Map<String,LoginInfo> infosBySessionId;
    private final Map<String,LoginInfo> infosByBearerToken;
    private TokenGenerator tokenGenerator;

    public LoginInfoCache() {

        this.infosBySessionId = new HashMap<String, LoginInfo>(1024);
        this.infosByBearerToken = new HashMap<String, LoginInfo>(1024);
    }

    public synchronized LoginInfo getLoginInfo(String sessionId) {

        LoginInfo ret = this.infosBySessionId.get(sessionId);

        return ret;
    }

    public synchronized LoginInfo getByBearerToken(String token) {

        LoginInfo ret = this.infosByBearerToken.get(token);

        return ret;
    }

    public synchronized LoginInfo removeLoginInfo(String sessionId) {

        return this.infosBySessionId.remove(sessionId);
    }

    private static Locale getLocale(OAuthPrincipal principal, Locale locale) {

        String locale_s = principal.getAdditionalAttribute("locale");

        if (locale_s != null) {

            Locale loc = LocaleHelper.localeFromXsLanguage(locale_s);

            if (loc != null && !loc.equals(locale)) {
                log.info("Replacing locale [{}] for principal [{}] with [{}] determined from OAuth properties.",
                        new Object[] {locale,principal.getName(),loc});
                return loc;
            }
        }

        return locale;
    }

    private static TimeZone getTimeZone(OAuthPrincipal principal, TimeZone timeZone) {

        String tz_s = principal.getAdditionalAttribute("zoneinfo");

        if (tz_s != null) {

            TimeZone tz = TimeZone.getTimeZone(tz_s);

            if (tz != null && !tz.equals(timeZone)) {
                log.info("Replacing timezone [{}] for principal [{}] with [{}] determined from OAuth properties.",
                        new Object[] {timeZone,principal.getName(),tz});
                return tz;
            }
        }

        return timeZone;
    }

    /**
     * Create a new session ID and cache the generated login info object.
     *
     * @param principal The parsed principal.
     * @param response The token response to use for performing refresh requests.
     * @param locale The locale as defined in the login frame.
     * @param timeZone The optional time zone as determined from the user agent.
     * @param maxAge The maximal lifetime of the generated session in milliseconds.
     * @param registerBearerToken Whether the given session information will be registered
     *                            by bearer token for using the session in requests
     *                            arriving with a bearer token but without a session ID.
     * @return The newly generated login info object.
     */
    public LoginInfo createLoginInfo(OAuthPrincipal principal, OAuthTokenResponse response,
            Locale locale, TimeZone timeZone,
            long maxAge, boolean registerBearerToken) {

        LoginInfo ret = null;

        timeZone = getTimeZone(principal,timeZone);
        locale = getLocale(principal,locale);

        int ntry = 0;
        do {
            ++ntry;

            String key = this.tokenGenerator.generateToken();

            ret = new LoginInfo(key,locale,timeZone);

            synchronized(this) {

                if (this.infosBySessionId.containsKey(key)) {
                    log.warn("Duplicate session ID generated by SecureRandom for principal [{}] of type [{}].",
                            principal.getName(),
                            principal.getClass().getName());

                    ret = null;
                }
                else {
                    this.infosBySessionId.put(key,ret);

                    if (registerBearerToken) {

                        if (response.getAccessToken() != null) {
                            LoginInfo oldInfo = this.infosByBearerToken.put(response.getAccessToken(),ret);

                            if (oldInfo != null) {

                                log.warn("Removing duplicate session entry [{}] for bearer token.",oldInfo.getSessionId());
                                this.infosBySessionId.remove(oldInfo.getSessionId());
                            }
                        }
                    }

                    ret.setCredentials(response, principal);
                }
            }

        } while (ret == null && ntry < 5);

        if (ret == null) {
            throw new SecurityException("["+ntry+
                    "] duplicate session IDs generated by SecureRandom for principal ["+principal.getName()+
                    "] of type ["+principal.getClass().getName()+"].");
        }

        ret.touch(maxAge);
        return ret;
    }

    public void gc() {

        long now = System.currentTimeMillis();

        synchronized (this) {

            if (log.isDebugEnabled()) {
                log.debug("Starting login info garbage collection, number of persisted session IDs is [{}]...",
                        LoginInfoCache.this.infosBySessionId.size());
            }

            Iterator<Map.Entry<String,LoginInfo>> it = LoginInfoCache.this.infosBySessionId.entrySet().iterator();

            while (it.hasNext()) {

                Entry<String, LoginInfo> e = it.next();

                if (e.getValue().getExpires() <= now) {

                    if (log.isWarnEnabled()) {

                        log.warn("Login [{}] expired without prior logout.",
                                e.getValue().getPrincipalsInfo());
                    }

                    it.remove();
                }
            }

            it = LoginInfoCache.this.infosByBearerToken.entrySet().iterator();

            while (it.hasNext()) {

                Entry<String, LoginInfo> e = it.next();

                if (e.getValue().getExpires() <= now) {

                    if (log.isWarnEnabled()) {

                        log.warn("Bearer Token [{}] for login [{}] expired without prior logout.",
                                e.getKey(),e.getValue().getPrincipalsInfo());
                    }

                    it.remove();
                }
            }

            if (log.isDebugEnabled()) {
                log.debug("Login info garbage collection finished, number of persisted session IDs is [{}].",
                        LoginInfoCache.this.infosBySessionId.size());
            }
        }
    }

    public void setTokenGenerator(TokenGenerator tokenGenerator) {
        this.tokenGenerator = tokenGenerator;
    }

}
