/***********************************************************
 * $Id$
 * 
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 19.09.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.oauth;

import java.util.Locale;
import java.util.TimeZone;
import java.util.concurrent.TimeoutException;

import org.clazzes.login.jbo.jwt.JWTokenClaims;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A cache entry for login credentials.
 */
public class LoginInfo {

    private static final Logger log = LoggerFactory.getLogger(LoginInfo.class);
    
    private final String sessionId;
    private OAuthPrincipal principal;
    private OAuthTokenResponse response;
    private Locale locale;
    private TimeZone timeZone;
    private long expires;
    private RefreshState refreshState;

    public LoginInfo(String sessionId, Locale locale, TimeZone timeZone) {
        this.sessionId = sessionId;
        this.locale = locale;
        this.timeZone = timeZone;
    }
    public String getSessionId() {
        return this.sessionId;
    }
    public synchronized OAuthPrincipal getPrincipal() {

        return this.principal;
    }

    public synchronized OAuthTokenResponse getResponse() {
        return this.response;
    }
    
    public synchronized void setCredentials(OAuthTokenResponse response, OAuthPrincipal principal) {
        this.response = response;
        this.principal = principal;
        
        if (this.response == null || this.principal == null) {
            this.refreshState = RefreshState.FAILED;
        }
        else {
            this.refreshState = RefreshState.CURRENT;
        }
        this.notifyAll();
    }
    
    public synchronized Locale getLocale() {
        return this.locale;
    }
    
    public synchronized TimeZone getTimeZone() {
        return this.timeZone;
    }
    
    /**
     * Check the validity of the tokens from the authentication provider and
     * return the original token response with a refresh token, if a refresh is needed.
     *  In this case, the refresh state is set to {@link RefreshState#UPDATING}.
     * 
     * @param timeout The timeout to wait for concurrent updates being performed.
     * @return The token response with a refresh token, if we need a refresh or
     *         <code>null</code>, if the tokens are still valid.
     * @throws TimeoutException If the refresh state is set to {@link RefreshState#FAILED}
     *              by a concurrent thread or we are unable to refresh the request by
     *              a refresh token.
     */
    public synchronized OAuthTokenResponse checkTokenValidity(long timeout) throws TimeoutException {
        
        if (this.refreshState == RefreshState.UPDATING) {
            
            try {
                this.wait(timeout);
                
            } catch (InterruptedException e) {
                throw new TimeoutException();
            }
        }

        if (this.refreshState != RefreshState.CURRENT) {
            throw new TimeoutException("Refresh of tokens for principal ["+this.principal.getName()+"] failed in a concurrent operation.");
        }
        
        boolean expired = false;
        long now = System.currentTimeMillis();
        
        if (this.response.getExpiresIn() != null) {
            
            long watermark = this.response.getTimestamp() + this.response.getExpiresIn().longValue();
            
            if (watermark <= now) {
                
                log.info("access token for principal [{}] expired by response timeout.",this.principal.getName());
                expired = true;
            }
        }
        
        if (!expired && this.principal.getIdToken() != null) {
            
            JWTokenClaims claims = this.principal.getIdToken().getClaimSet();
            
            if (claims.getExpiration() != null) {
                
                if (claims.getExpiration().longValue() <= now) {
                    
                    log.info("ID token for principal [{}] expired by expiration claim.",this.principal.getName());
                    expired = true;
                }
            }
        }
        
        if (expired) {
            
            if (this.response.getRefreshToken() == null) {
                
                throw new TimeoutException("ID token for principal ["+this.principal.getName()+"] expired without a refresh token.");
            }
            
            this.refreshState = RefreshState.UPDATING;
            return this.response;
        }
        
        return null;
    }
    
    public synchronized long getExpires() {
        return this.expires;
    }
    public synchronized void touch(long maxAge) {
        this.expires = System.currentTimeMillis() + maxAge;
    }
    public synchronized String getPrincipalsInfo() {

        if (this.principal == null) {
            return "<no principal>";
        }

        return this.principal.getName();
    }

}
