/***********************************************************
 * $Id$
 *
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 31.05.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.oauth;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.clazzes.login.jbo.json.JsonHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;

/**
 * A parse, which parses configuration objects, which are simple maps
 * of strings, numbers, booleans or string arrays.
 */
public class JsonConfigParser {

    private static final Logger log = LoggerFactory.getLogger(JsonConfigParser.class);

    /**
     * Parse a simple JSON configuration object.
     *
     * @param is A JSON input stream
     * @return A map with string keys and {@link String}, {@link Boolean},
     *         {@link Double}, arrays of strings or string->string maps.
     * @throws IOException On parse errors.
     */
    public static final Map<String,Object> parseConfig(InputStream is) throws IOException {

        JsonFactory jf = new JsonFactory();

        String ctxt  = "config";

        try (JsonParser reader = jf.createParser(is)) {

            Map<String,Object> ret = new HashMap<String,Object>();

            JsonHelper.beginObject(reader,ctxt);

            String name;

            while ((name = JsonHelper.nextName(reader,ctxt)) != null) {

                JsonToken tokenType = reader.nextToken();

                switch (tokenType) {

                case VALUE_NULL:
                     ret.put(name,null);
                    break;
                case VALUE_TRUE:
                case VALUE_FALSE:
                    ret.put(name,reader.getBooleanValue());
                    break;
                case VALUE_NUMBER_INT:
                case VALUE_NUMBER_FLOAT:
                    ret.put(name,reader.getDoubleValue());
                    break;
                case VALUE_STRING:
                    ret.put(name,reader.getText());
                    break;
                case START_ARRAY:

                    List<String> arr = new ArrayList<String>();

                    while (reader.nextToken().isScalarValue()) {
                        arr.add(reader.getText());
                    }

                    JsonHelper.endArray(reader,name);

                    ret.put(name,arr);
                    break;

                case START_OBJECT:

                    Map<String,String> map = new HashMap<String,String>();

                    String key;

                    while ((key = JsonHelper.nextName(reader, ctxt))!=null) {
                        String value = JsonHelper.nextString(reader,ctxt);
                        map.put(key,value);
                    }

                    JsonHelper.endObject(reader,ctxt);

                    ret.put(name,map);
                    break;

                default:
                    throw new IllegalArgumentException("Unsupported configuration member type ["+tokenType+"].");
                }
            }

            JsonHelper.endObject(reader,ctxt);

            return ret;
        }
    }

    /**
     * Parse a simple JSON string map object.
     *
     * @param is A JSON input stream
     * @return A map with string keys and string values.
     * @throws IOException On parse errors.
     */
    public static final Map<String,Object> parseStringMap(InputStream is) throws IOException {

        JsonFactory jf = new JsonFactory();

        String ctxt  = "config";

        try (JsonParser reader = jf.createParser(is)) {

            Map<String,Object> ret = new HashMap<String,Object>();

            JsonHelper.beginObject(reader,ctxt);

            String name;

            while ((name = JsonHelper.nextName(reader,ctxt)) != null) {

                JsonToken tokenType = reader.nextToken();

                switch (tokenType) {

                case VALUE_NULL:
                     ret.put(name,null);
                    break;
                case VALUE_TRUE:
                case VALUE_FALSE:
                    ret.put(name,reader.getBooleanValue());
                    break;
                case VALUE_NUMBER_INT:
                case VALUE_NUMBER_FLOAT:
                    ret.put(name,reader.getDoubleValue());
                    break;
                case VALUE_STRING:
                    ret.put(name,reader.getText());
                    break;
                case START_ARRAY:

                    List<String> arr = new ArrayList<String>();

                    while (reader.nextToken().isScalarValue()) {
                        arr.add(reader.getText());
                    }

                    JsonHelper.endArray(reader,name);

                    ret.put(name,arr);
                    break;

                default:
                    log.warn("Ignoring unknown property [{}] in string map response.",name);
                    reader.skipChildren();
                }
            }

            JsonHelper.endObject(reader,ctxt);

            return ret;
        }

    }

    public static final Map<String,String> parseAppUsers(String appUsers) {

        if (appUsers == null) {
            return null;
        }

        Map<String,String> ret = new HashMap<String,String>();

        String[] parts = appUsers.trim().split("\\s*,\\s*");

        for (String part:parts) {

            if (part.isEmpty()) {
                continue;
            }

            String[] appUser = part.split("\\s*:\\s*",2);

            if (appUser.length != 2) {
                throw new IllegalArgumentException("Invalid app user stanza ["+appUser+"] without colon.");
            }

            ret.put(appUser[0],appUser[1]);
        }

        return ret;
    }
}
