/***********************************************************
 * $Id$
 *
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 19.09.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.oauth;

import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;

import org.clazzes.util.sched.ITimedJob;
import org.clazzes.util.sched.impl.OneTimeSchedulerImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The garbage collector bean.
 */
public class GarbageCollector {

    private static final Logger log = LoggerFactory.getLogger(GarbageCollector.class);

    private LoginInfoCache loginInfoCache;
    private AuthStateCache authStateCache;

    private OneTimeSchedulerImpl oneTimeScheduler;
    private UUID jobId;

    private class GCCallable implements Callable<Void>, ITimedJob {

        @Override
        public Long getNextExecutionDelay() {

            return 60000L;
        }

        @Override
        public Void call() throws Exception {

            GarbageCollector.this.loginInfoCache.gc();
            GarbageCollector.this.authStateCache.gc();
            return null;
        }

    }

    public void executorServiceBound(ExecutorService executorService) {

        if (this.oneTimeScheduler == null) {
            this.oneTimeScheduler = new OneTimeSchedulerImpl();
            this.oneTimeScheduler.setExecutorService(executorService);
            this.oneTimeScheduler.start();

            this.jobId = this.oneTimeScheduler.scheduleJob(new GCCallable());

            log.info("Starting OAuth login garbage collector with job ID [{}].",this.jobId);
        }
    }

    public void executorServiceUnbound() {

        if (this.oneTimeScheduler != null) {

            log.info("Stopping OAuth login garbage collector with job ID [{}].",this.jobId);

            this.oneTimeScheduler.shutdownNow();
            this.oneTimeScheduler = null;
            this.jobId = null;
        }
    }

    public void setLoginInfoCache(LoginInfoCache loginInfoCache) {
        this.loginInfoCache = loginInfoCache;
    }

    public void setAuthStateCache(AuthStateCache authStateCache) {
        this.authStateCache = authStateCache;
    }
}
