/***********************************************************
 * $Id$
 *
 * http://www.clazzes.org
 *
 * Created: 02.04.2011
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.oauth;

import java.net.PasswordAuthentication;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Clock;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import org.clazzes.svc.api.ConfigWrapper;
import org.clazzes.svc.api.ConfigurationHelper;
import org.clazzes.svc.api.ServiceRegistry;
import org.clazzes.util.sec.OAuthCredentials;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigurationService implements Consumer<ConfigWrapper> {

    private static final Logger log = LoggerFactory.getLogger(ConfigurationService.class);

    private OAuthHttpClient oauthHttpClient;
    private TokenValidator tokenValidator;
    private final ServiceRegistry serviceRegistry;

    class DomainConfigRecord {
        private DomainManager domainManager;
        private String robotSupplierKey;

        public DomainConfigRecord(DomainManager domainManager,
                String robotSupplierKey) {
            this.domainManager = domainManager;
            this.robotSupplierKey = robotSupplierKey;
        }
        public DomainManager getDomainManager() {
            return this.domainManager;
        }
        public void setDomainManager(DomainManager domainManager) {
            this.domainManager = domainManager;
        }
        public String getRobotSupplierKey() {
            return this.robotSupplierKey;
        }
        public void setRobotSupplierKey(String robotSupplierKey) {
            this.robotSupplierKey = robotSupplierKey;
        }
    }

    /**
     * A mapping from domain names to domain configuration.
     */
    private final Map<String,DomainConfigRecord> domainConfigurations;
    private final List<String> guiDomains; // list of domains with user flow.

    public static final String CONFIG_PID = "org.clazzes.login.oauth";

    public ConfigurationService(ServiceRegistry serviceRegistry) {
        this.serviceRegistry = serviceRegistry;
        this.domainConfigurations = new HashMap<String,DomainConfigRecord>();
        this.guiDomains = new ArrayList<String>();
    }

    private static final String makeRobotSupplierKey(String domain) {
        return "oauth.robot.token."+domain;
    }

    private void stopOpenIdJobs() {

        for (Map.Entry<String,DomainConfigRecord> e : this.domainConfigurations.entrySet()) {

            String robotKey = e.getValue().getRobotSupplierKey();

            if (robotKey != null) {
                if (log.isDebugEnabled()) {
                    log.debug("Unregistering robot token supplier for domain [{}] with key [{}]",
                              e.getKey(),robotKey);
                }

                this.serviceRegistry.removeService(robotKey,Supplier.class);

                e.getValue().setRobotSupplierKey(null);
            }
        }
    }

    private void startOpenIdJobs() {
        for (Map.Entry<String,DomainConfigRecord> e : this.domainConfigurations.entrySet()) {
            DomainManager domainManager = e.getValue().getDomainManager();

            if (domainManager.getDomainConfig().isRobot()) {

                Supplier<? extends OAuthCredentials> supplier = domainManager.getCredentialsSupplier();

                String robotKey = makeRobotSupplierKey(e.getKey());

                if (log.isDebugEnabled()) {
                    log.debug("Registering robot token supplier for domain [{}] with key [{}]",
                              e.getKey(),robotKey);
                }

                this.serviceRegistry.addService(robotKey,Supplier.class,supplier);
            }
        }
    }

    private static URI getUri(ConfigWrapper properties, String key) {
        return properties.getParsed(URI::create,key);
    }

	private URI getFaviconLocation(ConfigWrapper domainProperties, URI configurationLocation,
			URI authorizationLocation) {

		URI faviconLocation = getUri(domainProperties, "faviconLocation");

        if (faviconLocation == null) {
            try {
                if (configurationLocation != null) {
                    faviconLocation = new URI(configurationLocation.getScheme(),null,
                    configurationLocation.getHost(),configurationLocation.getPort(),"/favicon.ico",null,null);
                }
                else if (authorizationLocation != null) {
                    faviconLocation = new URI(authorizationLocation.getScheme(),null,
                    authorizationLocation.getHost(),authorizationLocation.getPort(),"/favicon.ico",null,null);
                }

            } catch (URISyntaxException e) {
                throw new IllegalArgumentException("Couldn't guess the faviconLocation from the configurationLocation or the authoizationLocation, please specify it explicitly.",e);
            }
        }

		return faviconLocation;
	}

    private static PasswordAuthentication getPWAuth(String lbl, String user, String password) {

        if (user != null && password == null) {
            log.warn("No password for OAuth user role [{}] given, authentication will not work.",lbl);
        }

        if (user == null || password ==  null) {
            return null;
        }

        return new PasswordAuthentication(user,password.toCharArray());
    }

    private DomainConfig extractConfig(String domain, ConfigWrapper domainProperties)  {
        String label = domainProperties.getMandatoryString("label");
        String clientId = domainProperties.getString("clientId");
        String clientpw = domainProperties.getString("clientPassword");

        PasswordAuthentication clientAuth = getPWAuth("client",clientId,clientpw);

        RobotGrantType robotGrantType =
            domainProperties.getEnum(RobotGrantType.class,"robotGrantType");

        String robotClientId = domainProperties.getString("robotClientId");
        String robotClientPassword = domainProperties.getString("robotClientPassword");

        PasswordAuthentication robotAuth = robotGrantType != null && robotClientId != null
            ? getPWAuth("robotClient",robotClientId,robotClientPassword)
            : null;

        String robotUserId = domainProperties.getString("robotUserId");
        String robotUserPassword = domainProperties.getString("robotUserPassword");

        PasswordAuthentication robotUserCredentials = robotGrantType == RobotGrantType.password && robotUserId != null
            ? getPWAuth("robotUser",robotUserId,robotUserPassword)
            : null;

        URI configurationLocation = getUri(domainProperties, "configurationLocation");
        URI authorizationLocation = getUri(domainProperties, "authorizationLocation");

        URI tokenLocation = getUri(domainProperties, "tokenLocation");
        URI userLocation = getUri(domainProperties, "userLocation");

        URI faviconLocation = getFaviconLocation(domainProperties, configurationLocation, authorizationLocation);

        String scope = domainProperties.getString("scope");
        String prompt = domainProperties.getString("prompt");
        String accessType = domainProperties.getString("accessType");
        String resource = domainProperties.getString("resource");
        String groupRoleResource = domainProperties.getString("groupRoleResource");
        String optionsString = domainProperties.getString("options");
        EnumSet<ConfigOptions> options = ConfigOptions.parseOptions(optionsString);

        Map<String, String> appUsers = JsonConfigParser.parseAppUsers(domainProperties.getString("appUsers"));
        String robotScope = domainProperties.getString("robotScope");


        if (robotAuth == null && clientAuth == null) {
            throw new IllegalArgumentException("Neither a user nor a robot ClientID has been specified.");
        }

        if (authorizationLocation == null && configurationLocation == null && clientAuth != null) {
            // we need an authorization location for user flows.
            throw new IllegalArgumentException(
            "Neither an authorization URI nor a configuration URI has been given.");
        }

        if (userLocation == null && configurationLocation == null && clientAuth != null) {
            // we need a userinfo endpoint for GUI flows.
            throw new IllegalArgumentException(
            "Neither a user URI nor a configuration URI has been given.");
        }

        // A token endpoint is required for robot and interactive user cases.
        if (tokenLocation == null && configurationLocation == null) {
            throw new IllegalArgumentException(
            "Neither a token URI nor a configuration URI has been given.");
        }

        if (log.isDebugEnabled()) {
            if (clientAuth != null) {
                log.debug("Setting bind credentials for domain [{}] to [{}].",
                          domain,clientAuth.getUserName());
            }
            if (robotAuth != null) {
                log.debug("Setting robot bind credentials for domain [{}] to [{}] with grant type [{}].",
                          domain,robotAuth.getUserName(),robotGrantType);
            }
            if (robotUserCredentials != null) {
                log.debug("Setting robot user credentials for domain [{}] to [{}].",
                          domain,robotUserCredentials.getUserName());
            }
            log.debug("Setting configuration location for domain [{}] to [{}].",domain,configurationLocation);
            log.debug("Setting authorization location for domain [{}] to [{}].",domain,authorizationLocation);
            log.debug("Setting favicon location for domain [{}] to [{}].",domain,faviconLocation);
            log.debug("Setting token location for domain [{}] to [{}].",domain,tokenLocation);
            log.debug("Setting user location for domain [{}] to [{}].",domain,userLocation);
            log.debug("Setting scope for domain [{}] to [{}].",domain,scope);
            log.debug("Setting prompt for domain [{}] to [{}].",domain,prompt);
            log.debug("Setting accessType for domain [{}] to [{}].",domain,accessType);
            log.debug("Setting resource for domain [{}] to [{}].",domain,resource);
            log.debug("Setting groupRoleResource for domain [{}] to [{}].",domain,groupRoleResource);
            log.debug("Setting options for domain [{}] to [{}].",domain,options);
            log.debug("Setting appUsers for domain [{}] to [{}].",domain,appUsers);
            log.debug("Setting robot scope for domain [{}] to [{}].",
            domain,robotScope);
        }

        return new DomainConfig(
            domain,
            label,
            authorizationLocation,
            tokenLocation,
            userLocation,
            configurationLocation,
            faviconLocation,
            clientAuth,
            scope,
            prompt,
            accessType,
            resource,
            groupRoleResource,
            options,
            appUsers,
            robotGrantType,
            robotScope,
            robotAuth,
            robotUserCredentials
        );
    }



    @Override
    public void accept(ConfigWrapper t) {

        this.stopOpenIdJobs();
        this.domainConfigurations.clear();
        this.guiDomains.clear();

        ConfigWrapper domains = t.getSubTree("domain");

        if (domains == null) {
            log.warn("No domains configured in PID [{}], OAuth is deactivated.",CONFIG_PID);
            return;
        }

        for (String domain: domains.keySet()) {

            ConfigWrapper domainProperties =
                domains.getMandatorySubTree(domain);

            log.info("Extracting configuration for OAuth domain [{}]...",domain);

            if (log.isDebugEnabled()) {
                log.debug("Parsing domain properties [{}]",domainProperties);
            }

            DomainConfig config = extractConfig(domain, domainProperties);

            log.info("Final OAuth domain config is [{}].",config);

            this.domainConfigurations.put(domain,
                new DomainConfigRecord(
                    new DomainManager(
                            config, Clock.systemDefaultZone(),
                            this.oauthHttpClient, this.tokenValidator),
                    null));

            if (config.getClientCredentials() != null) {
                this.guiDomains.add(domain);
            }
        }

        this.startOpenIdJobs();
    }

    /**
     * @return the list of domains with a user flow for which we have a domain config.
     */
    public synchronized List<String> getDomains() {

        return new ArrayList<String>(this.guiDomains);
    }

    /**
     * @param domain A configured domain name.
     * @return The domain configuration for the given domain.
     */
    public synchronized DomainManager getDomainManager(String domain) {
        return Optional.ofNullable(this.domainConfigurations.get(domain))
            .map(r -> r.getDomainManager())
            .orElse(null);
    }

    public void setOauthHttpClient(OAuthHttpClient oauthHttpClient) {
        this.oauthHttpClient = oauthHttpClient;
    }

    public void setTokenValidator(TokenValidator tokenValidator) {
        this.tokenValidator = tokenValidator;
    }
}
