/***********************************************************
 * $Id$
 * 
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 19.09.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.oauth;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A cache for authentication states.
 */
public class AuthStateCache {

    private static final Logger log = LoggerFactory.getLogger(AuthStateCache.class);

    private final Map<String,AuthState> authStates;
    private TokenGenerator tokenGenerator;

    public AuthStateCache() {

        this.authStates = new HashMap<String, AuthState>(1024);

    }

    public synchronized AuthState getAuthState(String state) {

        AuthState ret = this.authStates.get(state);

        return ret;
    }

    public synchronized AuthState removeAuthState(String state) {

        return this.authStates.remove(state);
    }

    public AuthState createAuthState(Locale locale, long maxAge) {

        AuthState ret = null;

        int ntry = 0;
        do {
            ++ntry;

            String key = this.tokenGenerator.generateHexState();

            synchronized(this) {

                if (this.authStates.containsKey(key)) {
                    log.warn("Duplicate auth state generated by SecureRandom .");

                    ret = null;
                }
                else {
                    
                    ret = new AuthState(key,System.currentTimeMillis() + maxAge,locale);
                    this.authStates.put(key,ret);
                }
            }

        } while (ret == null && ntry < 5);

        if (ret == null) {
            throw new SecurityException("["+ntry+
                    "] auth states generated by SecureRandom.");
        }

        return ret;
    }

    public void gc() {

        long now = System.currentTimeMillis();

        synchronized (this) {

            if (log.isDebugEnabled()) {
                log.debug("Starting auth state garbage collection, number of persisted states is [{}]...",
                        AuthStateCache.this.authStates.size());
            }

            Iterator<Map.Entry<String,AuthState>> it = AuthStateCache.this.authStates.entrySet().iterator();

            while (it.hasNext()) {

                Entry<String, AuthState> e = it.next();

                if (e.getValue().getExpires() <= now) {

                    if (log.isWarnEnabled()) {

                        log.warn("auth state [{}] expired without prior logout.",
                                e.getValue().getState());
                    }

                    it.remove();
                }
            }

            if (log.isDebugEnabled()) {
                log.debug("Login info garbage collection finished, number of persisted states is [{}].",
                        AuthStateCache.this.authStates.size());
            }
        }
    }

    public void setTokenGenerator(TokenGenerator tokenGenerator) {
        this.tokenGenerator = tokenGenerator;
    }

}
