/***********************************************************
 * $Id$
 *
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 02.06.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.oauth;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.time.Clock;
import java.util.Arrays;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.stream.JsonReader;

/**
 * Parser for {@link OAuthTokenResponse} and {@link OAuthTokenErrorResponse}
 * objects.
 */
public class OAuthTokenResponseParser {

    private static final Logger log = LoggerFactory.getLogger(OAuthTokenResponseParser.class);

    static private OAuthTokenErrorResponse parseErrorInternal(JsonReader reader, String name) throws IOException {

        String error=null;
        String errorDescription=null;
        String errorUri=null;
        String timestamp=null;
        String traceId=null;
        String correlationId=null;
        long[] errorCodes=null;

        while (reader.hasNext()) {

            if (name == null) {
                name = reader.nextName();
            }

            if ("error".equals(name)) {
                error = reader.nextString();
            }
            else if ("error_description".equals(name)) {
                errorDescription = reader.nextString();
            }
            else if ("error_uri".equals(name)) {
                errorUri = reader.nextString();
            }
            else if ("timestamp".equals(name)) {
                timestamp = reader.nextString();
            }
            else if ("trace_id".equals(name)) {
                traceId = reader.nextString();
            }
            else if ("correlation_id".equals(name)) {
                correlationId = reader.nextString();
            }
            else if ("error_codes".equals(name)) {

                long[] tmp = new long[32];
                int ncodes = 0;

                reader.beginArray();

                while (reader.hasNext()) {

                    long code = reader.nextLong();

                    if (ncodes >= tmp.length) {
                        log.warn("Ignoring additonal error code [{}] (only 32 error codes are accepted in OAuth token error responses).",code);
                    }
                    else {
                        tmp[ncodes] = code;
                    }

                    ++ncodes;
                }

                reader.endArray();

                errorCodes = Arrays.copyOf(tmp,ncodes);
            }
            else {
                log.warn("Ignoring unknown property [{}] in OAuth token response.",name);
                reader.skipValue();
            }

            name = null;
        }

        reader.endObject();

        return new OAuthTokenErrorResponse(error,errorDescription,errorUri,timestamp,traceId,correlationId,errorCodes);
    }

    /**
     * Parse an OAuth token response.
     * @param is An input stream to read from.
     * @param redirectUri The original redirect URI to be stored for later refreshes of the response.
     * @param state The original state to be stored for later refreshes of the response.
     * @param refreshToken The original refresh token to be stored for later reuse, if the response from
     *                     the server did not contain a new refresh token.
     * @return The parsed token response.
     * @throws OAuthTokenErrorResponse
     */
    public static OAuthTokenResponse parseResponse(InputStream is,String redirectUri, String state, String refreshToken, Clock clock) throws IOException, OAuthTokenErrorResponse {

        try (JsonReader reader = new JsonReader(new InputStreamReader(is,"UTF-8"))) {

            String accessToken = null;
            String tokenType = null;
            Long expiresIn = null;
            String scope = null;
            String idToken = null;

            reader.beginObject();

            while (reader.hasNext()) {

                String name = reader.nextName();

                if ("error".equals(name) || "error_description".equals(name)) {
                    throw parseErrorInternal(reader,name);
                }
                else if ("access_token".equals(name)) {
                    accessToken = reader.nextString();
                }
                else if ("token_type".equals(name)) {
                    tokenType = reader.nextString();
                }
                else if ("expires_in".equals(name)) {
                    expiresIn = reader.nextLong() * 1000L;
                }
                else if ("scope".equals(name)) {
                    scope = reader.nextString();
                }
                else if ("refresh_token".equals(name)) {
                    refreshToken = reader.nextString();
                }
                else if ("id_token".equals(name)) {
                    idToken = reader.nextString();
                }
                else {
                    log.warn("Ignoring unknown property [{}] in OAuth token response.",name);
                    reader.skipValue();
                }
            }

            reader.endObject();

            return new OAuthTokenResponse(accessToken,tokenType,expiresIn,scope,refreshToken,idToken,redirectUri,state, clock);
        }
    }

    /**
     * Parse an OAuth token error response.
     * @param is An input stream to read from.
     * @return The parsed error response.
     */
    public static OAuthTokenErrorResponse parseErrorResponse(InputStream is) throws IOException {

        try (JsonReader reader = new JsonReader(new InputStreamReader(is,"UTF-8"))) {

            reader.beginObject();

            return parseErrorInternal(reader,null);
        }
    }

}
