/***********************************************************
 * $Id$
 *
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 19.09.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/
package org.clazzes.login.oauth;

import java.io.IOException;
import java.net.PasswordAuthentication;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.util.Locale;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import org.clazzes.login.oauth.i18n.OAuthMessages;
import org.clazzes.util.aop.i18n.Messages;
import org.clazzes.util.http.LocaleHelper;
import org.clazzes.util.http.RequestHelper;
import org.clazzes.util.http.UrlHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>The start servlet, which actually generates a state and redirects to
 * the authentication endpoint of the given domain.</p>
 */
public class OAuthStartServlet extends OAuthAbstrServlet {

    private static final long serialVersionUID = 6376913713678650071L;

    private static final Logger log = LoggerFactory.getLogger(OAuthStartServlet.class);

    protected void writeRedirect(Messages i18n,
            HttpServletResponse resp,
            String css,
            String domain,
            String authorizationUri,
            String state) throws  IOException, ServletException {

        try {

            String lang = LocaleHelper.toXsLanguage(i18n.getLocale());

            resp.setHeader("X-Frame-Options","SAMEORIGIN");
            resp.setHeader("Content-Language",lang);
            resp.setHeader("Cache-Control","no-cache");
            resp.setHeader("Pragma","no-cache");
            resp.setHeader("Expires","0");
            resp.setContentType("application/xhtml+xml");

            resp.getOutputStream().write("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" \"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n".getBytes("UTF-8"));

            XMLStreamWriter xsw = xmlOutputFactory.createXMLStreamWriter(resp.getOutputStream(),"UTF-8");

            xsw.setDefaultNamespace(XHTML_NS_URI);

            //xsw.writeStartDocument();

            xsw.writeStartElement("html");
            xsw.writeDefaultNamespace(XHTML_NS_URI);
            xsw.writeAttribute("lang",lang);
            xsw.writeAttribute("xml:lang",lang);
            xsw.writeStartElement("head");

            xsw.writeEmptyElement("meta");
            xsw.writeAttribute("http-equiv","Content-Type");
            xsw.writeAttribute("content","application/xhtml+xml");

            xsw.writeEmptyElement("link");
            xsw.writeAttribute("type","text/css");
            xsw.writeAttribute("rel","stylesheet");
            xsw.writeAttribute("href",css == null ? "oauth-login.css" : css);

            xsw.writeStartElement("script");
            xsw.writeCharacters("\nvar reloadUri = function() { window.location.href='"+authorizationUri+"';};\n");
            xsw.writeCharacters("\nsetTimeout(reloadUri,100);\n");
            xsw.writeEndElement();

            xsw.writeStartElement("title");
            xsw.writeCharacters("OAuth Single-Sign-On");
            xsw.writeEndElement(); // </title>

            xsw.writeEndElement(); // </head>

            xsw.writeStartElement("body");

            // hidden form
            xsw.writeStartElement("form");
            xsw.writeAttribute("id","loginStateForm");

            xsw.writeEmptyElement("input");
            xsw.writeAttribute("type","hidden");
            xsw.writeAttribute("name","state");
            xsw.writeAttribute("value",state);

            xsw.writeEndElement(); // </form>

            xsw.writeStartElement("p");

            xsw.writeCharacters(i18n.formatString("redirecting-to",domain));

            xsw.writeEndElement(); // </p>

            xsw.writeEndElement(); // </body>

            xsw.writeEndElement(); // </html>

            xsw.writeEndDocument();
            xsw.close();

            resp.flushBuffer();

        } catch (XMLStreamException e) {

            throw new ServletException("Error setting XML stream writer",e);
        }

    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException {

        String pi = req.getPathInfo();

        URI requestURI;

        try {
            requestURI = RequestHelper.getOriginalRequestUri(req);
        } catch (ParseException | URISyntaxException e) {
           throw new ServletException("Unable to parse full request URI for request to ["+RequestHelper.getRequestUrl(req)+"]",e);
        }

        if (log.isDebugEnabled()) {
            log.debug("Received request to [{}]",requestURI);
        }

        if (pi == null) {

            // domain selection by user.
            String domain = req.getParameter("domain");

            // start authentication and generate new AuthState, if state is null
            DomainManager domainManager = this.configurationService.getDomainManager(domain);
            DomainConfig domainConfig = domainManager.getDomainConfig();

            if (domainConfig == null) {
                log.info("Received request to [{}] with invalid domain parameter.",requestURI);
                resp.sendError(HttpServletResponse.SC_BAD_REQUEST);
                return;
            }

            // associate domain to state.
            String state = req.getParameter("state");

            if (state == null) {
                log.info("Received request to [{}] without state parameter.",requestURI);
                resp.sendError(HttpServletResponse.SC_BAD_REQUEST);
                return;
            }

            AuthState authState = this.authStateCache.getAuthState(state);

            if (authState == null) {
                log.info("Received request to [{}] with invalid or expired state parameter.",requestURI);
                resp.sendError(HttpServletResponse.SC_BAD_REQUEST);
                return;
            }


            String css = req.getParameter("css");
            Locale locale = getRequestLocale(req);
            Messages i18n = OAuthMessages.getMesssages(locale);

            authState.init(domain,System.currentTimeMillis()+900000L);

            /*
            https://login.microsoftonline.com/{tenant}/oauth2/v2.0/authorize?
                client_id=6731de76-14a6-49ae-97bc-6eba6914391e
                &response_type=code
                &redirect_uri=http%3A%2F%2Flocalhost%2Fmyapp%2F
                &response_mode=query
                &scope=openid%20offline_access%20https%3A%2F%2Fgraph.microsoft.com%2Fmail.read
                &state=12345
            */

            URI au = domainConfig.getAuthorizationLocation();

            PasswordAuthentication pa = domainConfig.getClientCredentials();

            try {

                if (au == null) {

                    au = domainManager.getOpenIdLocation("authorization_endpoint");
                }

                if (au == null) {
                    throw new ServletException("No authorization URI given by OpenID configuration for domain ["+
                            domainConfig.getDomain()+"] or OpenID configuration not yet loaded.");
                }

                URI redirectUri = new URI(requestURI.getScheme(),requestURI.getUserInfo(),requestURI.getHost(),requestURI.getPort(),
                        this.oauthHttpLoginService.getRedirectUrl(),
                        null,
                        requestURI.getFragment());

                String authorizationUri = au.toString();

                if (domainConfig.getOptions().contains(ConfigOptions.reverseProxyIntegrated)) {

                    URI proxyAu = new URI(requestURI.getScheme(),requestURI.getUserInfo(),
                                          requestURI.getHost(),requestURI.getPort(),
                                          au.getPath(),
                                          au.getQuery(),
                                          au.getFragment());

                    log.info("Using proxied authorization URI [{}] instead of [{}]",
                             proxyAu,au);

                    authorizationUri = proxyAu.toString();
                }
                else {
                    authorizationUri = au.toString();
                }

                if (log.isDebugEnabled()) {
                    log.debug("Raw authorization URI is [{}].",authorizationUri);
                    log.debug("Raw redirect URI is [{}].",redirectUri);
                }

                authorizationUri = UrlHelper.appendQueryParameterToUrl(authorizationUri,"client_id",pa.getUserName());
                authorizationUri = UrlHelper.appendQueryParameterToUrl(authorizationUri,"response_type","code");
                authorizationUri = UrlHelper.appendQueryParameterToUrl(authorizationUri,"redirect_uri",redirectUri.toString());

                if (domainConfig.getScope() != null) {
                    authorizationUri = UrlHelper.appendQueryParameterToUrl(authorizationUri,"scope",domainConfig.getScope());
                }

                if (domainConfig.getPrompt() != null) {
                    authorizationUri = UrlHelper.appendQueryParameterToUrl(authorizationUri,"prompt",domainConfig.getPrompt());
                }

                if (domainConfig.getAccessType() != null) {
                    authorizationUri = UrlHelper.appendQueryParameterToUrl(authorizationUri,"access_type",domainConfig.getAccessType());
                }

                if (domainConfig.getResource() != null) {
                    authorizationUri = UrlHelper.appendQueryParameterToUrl(authorizationUri,"resource",domainConfig.getResource());
                }

                if (domainConfig.getOptions().contains(ConfigOptions.propagateLocale)) {
                    authorizationUri = UrlHelper.appendQueryParameterToUrl(authorizationUri,"locale",LocaleHelper.toXsLanguage(locale));
                }

                authorizationUri = UrlHelper.appendQueryParameterToUrl(authorizationUri,"state",authState.getState());

                if (log.isDebugEnabled()) {
                    log.debug("Using full authorization URI [{}].",authorizationUri);
                }

                this.writeRedirect(i18n,resp,css,authState.getDomain(),authorizationUri,authState.getState());

            } catch (URISyntaxException e) {

                throw new ServletException("Error assembling redirect or authorization URI",e);

            } catch (IllegalStateException e) {

                log.error("OpenID configuration of domain ["+domainConfig.getDomain()+"] not loaded while requesting authorization location",e);
                resp.sendError(HttpServletResponse.SC_BAD_REQUEST);
                return;
            }

        }
        else {
            resp.sendError(HttpServletResponse.SC_NOT_FOUND);
        }
    }

    @Override
    public String getServletInfo() {

        return OAuthStartServlet.class.getSimpleName();
    }

}
