/***********************************************************
 * $Id$
 *
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 19.09.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.oauth;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.Principal;
import java.time.Clock;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;
import java.util.concurrent.TimeoutException;

import javax.security.auth.login.LoginException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.clazzes.login.htpasswd.jaas.HtpasswdAuthServiceFactory;
import org.clazzes.login.htpasswd.jaas.HtpasswdGroup;
import org.clazzes.login.htpasswd.jaas.IHtpasswdAuthService;
import org.clazzes.login.jbo.jwt.JWToken;
import org.clazzes.login.oauth.i18n.OAuthMessages;
import org.clazzes.util.aop.ThreadLocalManager;
import org.clazzes.util.aop.i18n.Messages;
import org.clazzes.util.http.sec.HttpLoginService;
import org.clazzes.util.osgi.ServiceMap;
import org.clazzes.util.sec.DomainPasswordLoginService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>An implementation of a {@link HttpLoginService}, which provides
 * HTTP session management for a given {@link DomainPasswordLoginService}</p>
 *
 * <p>This service binds the HTTP locale to the current thread using
 * {@link ThreadLocalManager#bindLoginLocale(Locale)} in order to allow
 * {@link DomainPasswordLoginService} implementations to obtain the user's
 * language when called from a HTTP context.
 * </p>
 */
public class OAuthHttpLoginService implements HttpLoginService {

    private static final Logger log = LoggerFactory.getLogger(OAuthHttpLoginService.class);

    private static final String LOGIN_URL = "/oauth-login/login";
    private static final String REDIRECT_URL = "/oauth-login/auth";

    private int sessionTimeout;
    private String sessionCookie;
    private boolean secureCookie;
    private SameSitePolicy sameSitePolicy;
    private String alternativeMechanism;

    private String delegateDomain;
    private TokenType delegateTokenType;

    private LoginInfoCache loginInfoCache;

    private ConfigurationService configurationService;

    private OAuthHttpClient oauthHttpClient;

    private TokenValidator tokenValidator;

    private ServiceMap httpLoginServiceMap;

    public OAuthHttpLoginService() {
    }

    HttpLoginService getAlternativeService() {
        if (this.alternativeMechanism != null &&
            !this.alternativeMechanism.isBlank()&&
            !ConfigurationService.CONFIG_PID.equals(this.alternativeMechanism)) {
            return (HttpLoginService)this.httpLoginServiceMap.getService(this.alternativeMechanism);
        }
        else {
            return null;
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.http.sec.HttpLoginService#getLoginUrl()
     */
    @Override
    public String getLoginUrl() {

        return LOGIN_URL;
    }

    public String getRedirectUrl() {

        return REDIRECT_URL;
    }

    private final String parseCookie(HttpServletRequest req) {

        String allCookies_s = req.getHeader("Cookie");

        if (allCookies_s == null) {
            return null;
        }

        String[] allCookies = allCookies_s.split("\s*;\s*");

        for (String s: allCookies) {

            String[] kv = s.split("\s*=\s*",2);

            if (kv.length == 2 && kv[0].equals(this.sessionCookie)) {
                return kv[1];
            }
        }

        return null;
    }

    private LoginInfo getLoginInfoFromCookie(HttpServletRequest req) {

        String sessionId = this.parseCookie(req);

        if (sessionId == null) {

            if (this.delegateDomain == null || this.delegateDomain.isEmpty()) {
                return null;
            }
            else {
                Locale loc = req.getLocale();

                if (loc == null) {
                    // should be unreachable, but if a request implementation fails to return a Locale this at least fulfills this method's javadoc
                    loc = Locale.getDefault();
                }

                Messages i18n = OAuthMessages.getMesssages(loc);

                return this.getLoginInfoFromBearerToken(i18n, req);
            }
        }
        else {
            return this.loginInfoCache.getLoginInfo(sessionId);
        }
    }

    private static final String parseBearerToken(HttpServletRequest req) {

        String auth_s = req.getHeader("Authorization");

        if (auth_s == null) {
            return null;
        }

        auth_s = auth_s.trim();

        if (!auth_s.startsWith("Bearer ")) {
            return null;
        }

        return auth_s.substring(7).trim();
    }

    private LoginInfo getLoginInfoFromBearerToken(Messages i18n, HttpServletRequest req) {

        String token = OAuthHttpLoginService.parseBearerToken(req);

        if (token == null) {
            return null;
        }
        else {
            LoginInfo ret = this.loginInfoCache.getByBearerToken(token);

            if (ret != null) {
                return ret;
            }

            DomainConfig domainConfig = this.configurationService.getDomainManager(this.delegateDomain).getDomainConfig();

            try {

                String scope = domainConfig.getScope();

                OAuthTokenResponse response;

                if (this.delegateTokenType == TokenType.JWT) {

                    response = new OAuthTokenResponse(token,"jwt",null,scope,null,null,null,null, Clock.systemDefaultZone());
                }
                else {
                    response = new OAuthTokenResponse(token,"bearer",null,scope,null,null,null,null, Clock.systemDefaultZone());
                }

                OAuthPrincipal principal = parsePrincipal(this.delegateDomain, response, i18n);

                return this.loginInfoCache.createLoginInfo(principal,response,i18n.getLocale(),null,this.sessionTimeout*60000L,true);

            } catch (Exception e) {

                log.error("Validation of Bearer Token for domain ["+this.delegateDomain+"] failed",e);

                return null;
            }
        }
    }

    private OAuthTokenResponse refreshToken(Messages i18n, String domain, OAuthTokenResponse originalResponse) {

        DomainManager domainManager = this.configurationService.getDomainManager(domain);
        DomainConfig domainConfig = domainManager.getDomainConfig();

        URI tokenUri = null;

        try {
            tokenUri = domainManager.getTokenUri();

            String scope = domainConfig.getScope();

            log.info("Refreshing token from [{}] with redirect URI to [{}].",
                    tokenUri,originalResponse.getRedirectUri());


            OAuthTokenResponse tokenResponse = this.oauthHttpClient.refreshToken(tokenUri,
                    originalResponse.getRedirectUri(),
                    originalResponse.getState(),scope,domainConfig.getClientCredentials(),
                    originalResponse.getRefreshToken(), domainConfig.getOptions());

            return tokenResponse;

        } catch (Exception e) {

            log.error("Token refresh request to ["+tokenUri+"] failed",e);

            return null;
        }
    }

    public LoginInfo checkLoginInfo(HttpServletRequest req) {

        LoginInfo loginInfo = this.getLoginInfoFromCookie(req);

        if (loginInfo == null) {

           return null;
        }

        OAuthPrincipal ret = loginInfo.getPrincipal();

        if (ret == null) {
            return null;
        }

        try {
            OAuthTokenResponse originalResponse = loginInfo.checkTokenValidity(60000L);

            if (originalResponse != null) {

                Locale loc = null;

                if (loginInfo != null) {
                    loc = loginInfo.getLocale();
                }

                if (loc == null) {
                    loc = req.getLocale();
                }

                if (loc == null) {
                    // should be unreachable, but if a request implementation fails to return a Locale this at least fulfills this method's javadoc
                    loc = Locale.getDefault();
                }

                Messages i18n = OAuthMessages.getMesssages(loc);

                OAuthTokenResponse response = this.refreshToken(i18n,ret.getDomain(), originalResponse);

                OAuthPrincipal newPrincipal = null;

                if (response != null) {

                    try {
                        newPrincipal = this.parsePrincipal(ret.getDomain(),response,i18n);

                    } catch (OAuthTokenErrorResponse | IOException e) {

                        log.error("Cannot parse refreshed principal of original principal ["+ret.getName()+"], purging HTTP session.",e);
                    }

                    if (newPrincipal == null) {
                        response = null;
                    }
                }

                loginInfo.setCredentials(response,newPrincipal);

                ret = newPrincipal;

                if (ret == null) {
                    // purge HTTP session.
                    this.loginInfoCache.removeLoginInfo(loginInfo.getSessionId());
                }
            }

        } catch (TimeoutException e) {

            log.warn("Tokens for principal [{}] expired without a refresh URL.",ret.getName());
            return null;
        }

        if (ret != null) {
            loginInfo.touch(this.sessionTimeout * 60000L);
        }

        return loginInfo;
    }

    @Override
    public Principal checkLogin(HttpServletRequest req) {

        LoginInfo loginInfo = this.checkLoginInfo(req);

        if (loginInfo == null) {

            HttpLoginService alternativeService = this.getAlternativeService();

            if (alternativeService != null) {
                return alternativeService.checkLogin(req);
            }

            return null;
        }

        return loginInfo.getPrincipal();
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.http.sec.HttpLoginService#checkLoginGroups(javax.servlet.http.HttpServletRequest)
     */
    @Override
    public List<? extends Principal> checkLoginGroups(HttpServletRequest req) {

        LoginInfo loginInfo = this.getLoginInfoFromCookie(req);

        if (loginInfo != null) {
            return loginInfo.getPrincipal().getGroups();
        }

        HttpLoginService alternativeService = this.getAlternativeService();

        if (alternativeService != null) {
            return alternativeService.checkLoginGroups(req);
        }

        return null;
    }

    @Override
    public Locale getLocale(HttpServletRequest req) {

        Locale loc = null;
        LoginInfo loginInfo = this.getLoginInfoFromCookie(req);

        if (loginInfo != null) {
            loc = loginInfo.getLocale();
        }
        else {
            HttpLoginService alternativeService = this.getAlternativeService();

            if (alternativeService != null) {
                loc = alternativeService.getLocale(req);
            }
        }

        if (loc == null) {
            loc = req.getLocale();
        }

        if (loc == null) {
            // should be unreachable, but if a request implementation fails to return a Locale this at least fulfills this method's javadoc
            loc = Locale.getDefault();
        }

        return loc;
    }

    @Override
    public TimeZone getTimeZone(HttpServletRequest req) {

        LoginInfo loginInfo = this.getLoginInfoFromCookie(req);

        TimeZone tz = null;

        if (loginInfo != null) {
            tz = loginInfo.getTimeZone();
        }
        else {
            HttpLoginService alternativeService = this.getAlternativeService();

            if (alternativeService != null) {
                tz = alternativeService.getTimeZone(req);
            }
        }

        if (tz == null) {
            // This is most likely reached, when loginInfo.getTimeZone() returned null.
            tz = TimeZone.getDefault();
        }

        return tz;
    }

    @Override
    public boolean checkPermission(HttpServletRequest req, String context) {

        if (LOGIN_URL.equals(context)) {
            return true;
        }

        HttpLoginService alternativeService = this.getAlternativeService();

        if (alternativeService != null) {
            return alternativeService.checkPermission(req,context);
        }

        return false;
    }

    @Override
    public void logout(HttpServletRequest req) {

        String sessionId = this.parseCookie(req);

        if (sessionId == null) {
            return;
        }

        LoginInfo loginInfo = this.loginInfoCache.removeLoginInfo(sessionId);

        if (loginInfo != null) {

            log.info("OAuth Logout of [{}].",loginInfo.getPrincipalsInfo());

            DomainManager domainManager = this.configurationService.getDomainManager(loginInfo.getPrincipal().getDomain());
            DomainConfig domainConfig = domainManager.getDomainConfig();

            if (domainConfig != null &&
                domainConfig.getConfigurationLocation() != null &&
                !domainConfig.getOptions().contains(ConfigOptions.renderUserLogout)) {

                try {

                    URI logoutUri = domainManager.getOpenIdLocation("end_session_endpoint");

                    if (logoutUri != null &&loginInfo.getResponse().getAccessToken() != null) {
                        this.oauthHttpClient.logout(logoutUri,loginInfo.getResponse().getAccessToken());
                    }

                } catch (Exception e) {
                    log.warn("Unable to retrieve the end_session_endpoint URI for domain ["+loginInfo.getPrincipal().getDomain()+"]",e);
                }
            }

        }
        else {
            HttpLoginService alternativeService = this.getAlternativeService();

            if (alternativeService != null) {
                alternativeService.logout(req);
            }
        }
    }

    public List<String> getDomains() {
        return this.configurationService.getDomains();
    }

    private static final void fetchJaasInfo(String domain, String user, Map<String,Object> attributes) throws LoginException {

        IHtpasswdAuthService svc = HtpasswdAuthServiceFactory.getService(domain);

        List<HtpasswdGroup> htpasswdGroups = svc.getUserGroups(user);

        if (htpasswdGroups != null) {
            List<String> groups = new ArrayList<String>(htpasswdGroups.size());
            for (HtpasswdGroup grp:htpasswdGroups) {
                groups.add(grp.getName());
            }

            if (log.isDebugEnabled()) {
                log.debug("Enhancing OAuth principal [{}/{}] with JAAS groups {}.",
                          domain,user,groups);
            }

            attributes.put("groups",groups);
        }

        Map<String,String> htpasswdAtts = svc.getUserClaims(user);

        if (htpasswdAtts != null) {
            attributes.putAll(htpasswdAtts);

            if (log.isDebugEnabled()) {
                log.debug("Enhancing OAuth principal [{}/{}] with JAAS attributes {}.",
                          domain,user,htpasswdAtts);
            }
        }
    }

    public OAuthPrincipal parsePrincipal(String domain, OAuthTokenResponse response, Messages i18n) throws OAuthTokenErrorResponse, IOException {

        OAuthPrincipal ret = null;

        if (i18n.getLocale() != null) {
            ThreadLocalManager.bindLoginLocale(i18n.getLocale());
        }

        try {

            DomainManager domainManager = this.configurationService.getDomainManager(domain);
            DomainConfig domainConfig = domainManager.getDomainConfig();

            URI userLocation = domainConfig.getUserLocation();

            if (userLocation == null || "jwt".equals(response.getTokenType())) {

                JWToken token = this.tokenValidator.validateToken(domainManager,response,i18n);

                if (token != null) {

                    Map<String, String> appUsers = domainConfig.getAppUsers();

                    if (appUsers != null) {

                        String appid = (String)token.getClaimSet().getAdditionalClaim("appid");

                        if (appid != null) {
                            String user = appUsers.get(appid);

                            if (user != null) {
                                log.info("Enhancing token for appid [{}] with JAAS user [{}]",appid,user);

                                try {
                                    fetchJaasInfo(domain,user,token.getClaimSet().getAdditionalClaims());

                                    log.info("Enhanced appid claim set is [{}]",
                                             token.getClaimSet());

                                } catch (LoginException e) {
                                    log.error("Error enhancing token for appid ["+appid+"] with JAAS user ["+user+"]",e);
                                    throw new OAuthTokenErrorResponse("openid-configuration-invalid",i18n);
                                }
                            }
                        }
                    }

                    ret = new OAuthPrincipal(domain,token,response.getAccessToken(),domainConfig.getGroupRoleResource());
                }
                else if (response.getAccessToken() == null) {

                    // This case should be impossible, because validateToken only returns null,
                    // if do not have an ID Token.
                    log.error("JWT Bearer Token for ["+domainConfig.getDomain()+"] was empty and no access token given.");
                    throw new OAuthTokenErrorResponse("openid-token-validation-failed",i18n);
                }
                else {
                    try {
                        userLocation = domainManager.getOpenIdLocation("userinfo_endpoint");

                    } catch (URISyntaxException e) {

                        log.error("OpenID configuration of domain ["+domainConfig.getDomain()+"] contains and invalid user location",e);
                        throw new OAuthTokenErrorResponse("openid-configuration-invalid",i18n);

                    } catch (IllegalStateException e) {

                        log.error("OpenID configuration of domain ["+domainConfig.getDomain()+"] not loaded while requesting user location",e);
                        throw new OAuthTokenErrorResponse("openid-configuration-not-loaded",i18n);
                    }
                }
            }

            if (ret == null) {
                if (userLocation == null) {
                    log.error("No ID token given and OpenID configuration of domain ["+domainConfig.getDomain()+"] contains no user location");
                    throw new OAuthTokenErrorResponse("openid-configuration-invalid",i18n);
                }

                Map<String,Object> attributes = this.oauthHttpClient.loadStringMap(userLocation,response.getAccessToken());

                ret = new OAuthPrincipal(domain,attributes,response.getAccessToken(),domainConfig.getGroupRoleResource());
            }
        }
        finally {
            if (i18n.getLocale() != null) {
                ThreadLocalManager.unbindLoginLocale();
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Parsed OAuth principal as [{}]",ret);
        }

        return ret;
    }

    public OAuthPrincipal tryLogin(HttpServletRequest req, HttpServletResponse resp, String domain,
           OAuthTokenResponse response, Messages i18n) throws IOException, OAuthTokenErrorResponse {

        OAuthPrincipal principal = this.parsePrincipal(domain, response, i18n);
        TimeZone tz = null;

        LoginInfo loginInfo = this.loginInfoCache.createLoginInfo(principal,response,
                i18n.getLocale(),null,this.sessionTimeout * 60000L,
                domain.equals(this.delegateDomain));

        log.info("Successful login of user [{}] with groups [{}] locale [{}] and timezone [{}].",
                new Object[]{principal.getName(),principal.getGroups(),i18n.getLocale(),tz == null ? null : tz.getID()});

        StringBuffer cookie = new StringBuffer();

        //
        // from http://tools.ietf.org/html/rfc6265
        //
        // SID=31d4d96e407aad42; Path=/; Secure; HttpOnly

        cookie.append(this.sessionCookie);
        cookie.append("=");
        cookie.append(loginInfo.getSessionId());

        cookie.append("; Path=/; ");

        if (this.sameSitePolicy != null) {
            cookie.append("SameSite=");
            cookie.append(this.sameSitePolicy);
            cookie.append("; ");
        }

        if (this.secureCookie) {
            cookie.append("Secure; ");
        }

        cookie.append("HttpOnly");

        resp.setHeader("Set-Cookie",cookie.toString());

        return principal;
    }

    public LoginInfoCache getLoginInfoCache() {
        return this.loginInfoCache;
    }

    public void setLoginInfoCache(LoginInfoCache loginInfoCache) {
        this.loginInfoCache = loginInfoCache;
    }

    public OAuthHttpClient getOauthHttpClient() {
        return this.oauthHttpClient;
    }

    public void setOauthHttpClient(OAuthHttpClient oauthHttpClient) {
        this.oauthHttpClient = oauthHttpClient;
    }

    public void setSessionTimeout(int sessionTimeout) {
        this.sessionTimeout = sessionTimeout;
    }

    public int getSessionTimeout() {
        return this.sessionTimeout;
    }

    public void setSessionCookie(String sessionCookie) {
        this.sessionCookie = sessionCookie;
    }

    public void setSecureCookie(boolean secureCookie) {
        this.secureCookie = secureCookie;
    }

    public void setSameSitePolicyString(String ssp) {

        String v = ssp.trim();

        this.sameSitePolicy = v.isEmpty() ? null : SameSitePolicy.valueOf(v);
    }

    public String getDelegateDomain() {
        return this.delegateDomain;
    }

    /**
     * @param delegateDomain Set the domain to which requests with an incoming
     *          bearer token without a session cookie will be delegated.
     */
    public void setDelegateDomain(String delegateDomain) {
        this.delegateDomain = delegateDomain;
    }

    /**
     * @param delegateTokenType The type of bearer token expected from
     *                          the delegate doamin.
     */
    public void setDelegateTokenType(TokenType delegateTokenType) {
        this.delegateTokenType = delegateTokenType;
    }

    public void setAlternativeMechanism(String alternativeMechanism) {
        this.alternativeMechanism = alternativeMechanism;
    }

    public void setConfigurationService(
            ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    public void setTokenValidator(TokenValidator tokenValidator) {
        this.tokenValidator = tokenValidator;
    }

    public void setHttpLoginServiceMap(ServiceMap httpLoginServiceMap) {
        this.httpLoginServiceMap = httpLoginServiceMap;
    }

}
