/***********************************************************
 * $Id$
 *
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 31.05.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.oauth;

import java.io.IOException;
import java.net.PasswordAuthentication;
import java.net.URI;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;

import org.clazzes.login.jbo.jwt.JWKPubKey;
import org.clazzes.login.jbo.jwt.JWKeyParser;

/**
 * The interface for issuing HTTP requests to OAuth services.
 */
public interface OAuthHttpClient {

    /**
     * Load a JSON config from an URL.
     *
     * @param url A URL returning a flat JSON object.
     * @return A parsed configuration as returned by {@link JsonConfigParser#parseConfig(java.io.InputStream)}
     * @throws IOException
     */
    public Map<String,Object> loadConfiguration(URI url) throws IOException;

    /**
     * Load a JSON string map from an URL.
     *
     * @param url A URL returning a flat JSON object.
     * @param bearerToken An optional bearer token used to authenticate against the server.
     * @return A parsed strgin map as returned by {@link JsonConfigParser#parseStringMap(java.io.InputStream)}
     * @throws IOException
     */
    public Map<String,Object> loadStringMap(URI url, String bearerToken) throws IOException;

    /**
     * Load a list of public keys.
     *
     * @param url The URL to load a JWK public key list from.
     * @return The parsed list of JWK public keys as returned by
     *         {@link JWKeyParser#parsePubKeyList(java.io.InputStream)}.
     */
    public List<JWKPubKey> loadPublicKeys(URI url) throws IOException;

    /**
     * Request a token after receiving an authorization code.
     * @param url The URL of the token endpoint.
     * @param redirectUri The redirect URI as specified during the authorization handshake.
     * @param state The application-generated random state.
     * @param scope The token scope.
     * @param clientCredentials The registered client ID and secret of our application.
     * @param authCode The authorization code fetched during the authorization handshake.
     * @param options Domain options to determin whetehr to pass client credentials as basic auth header.
     * @return A token response.
     *
     */
    public OAuthTokenResponse requestToken(URI url, String redirectUri, String state, String scope, PasswordAuthentication clientCredentials, String authCode, EnumSet<ConfigOptions> options) throws IOException, OAuthTokenErrorResponse;

    /**
     * Request a token after receiving an authorization code.
     * @param url The URL of the token endpoint.
     * @param redirectUri The redirect URI as specified during the authorization handshake.
     * @param state The application-generated random state.
     * @param scope The token scope.
     * @param clientCredentials The registered client ID and secret of our application.
     * @param refreshToken The refresh token previously received by {@link #requestToken(URI, String, String, String, PasswordAuthentication, String, EnumSet)}
     * @param options Domain options to determin whetehr to pass client credentials as basic auth header.
     * @return A token response.
     *
     */
    public OAuthTokenResponse refreshToken(URI url, String redirectUri, String state, String scope, PasswordAuthentication clientCredentials, String refreshToken, EnumSet<ConfigOptions> options) throws IOException, OAuthTokenErrorResponse;

    /**
     * Perform a logout on an end session URL.
     *
     * @param url A URL returning either a redirect or some plain text.
     * @param bearerToken An access token referring to the session to log out.
     * @throws IOException
     */
    public void logout(URI url, String bearerToken) throws IOException;

    /**
     * Request a robot token using client crendentials flow.
     * @param url The URL of the token endpoint.
     * @param scope The token scope.
     * @param clientCredentials The registered robot client ID and secret of our application.
     * @param options Domain options to determin whetehr to pass client credentials as basic auth header.
     * @return A token response.
     *
     */
    public OAuthTokenResponse requestRobotToken(URI url, String scope, PasswordAuthentication clientCredentials, EnumSet<ConfigOptions> options) throws IOException, OAuthTokenErrorResponse;

    /**
     * Request a robot token using password flow.
     * @param url The URL of the token endpoint.
     * @param scope The token scope.
     * @param clientCredentials The registered robot client ID and secret of our application.
     * @param resourceOwnerCredentials The resource owner user/password.
     * @param options Domain options to determin whetehr to pass client credentials as basic auth header.
     * @return A token response.
     *
     */
    public OAuthTokenResponse requestPasswordFlowToken(URI url, String scope, PasswordAuthentication clientCredentials, PasswordAuthentication resourceOwnerCredentials, EnumSet<ConfigOptions> options) throws IOException, OAuthTokenErrorResponse;
}
