/***********************************************************
 * $Id$
 *
 * HTTP Login service adapter of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 19.09.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/
package org.clazzes.login.oauth;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.util.Locale;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import org.clazzes.login.oauth.i18n.OAuthMessages;
import org.clazzes.util.aop.i18n.Messages;
import org.clazzes.util.http.LocaleHelper;
import org.clazzes.util.http.RequestHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>The login servlet, which actually provides for the login form according
 * to the documentation for {@link OAuthHttpLoginService#getRedirectUrl()}.</p>
 *
 * <p>When accepting a GET request, the login form is simply rendered based
 * on the login status of the current HTTP session.</p>
 */
public class OAuthAuthServlet extends OAuthAbstrServlet {

    private static final long serialVersionUID = 6376913713678650071L;

    private static final Logger log = LoggerFactory.getLogger(OAuthAuthServlet.class);


    protected void renderAuthenticationResult(Messages i18n,
            HttpServletResponse resp,
            String css,
            String domain,
            String scope,
            OAuthTokenErrorResponse error
            ) throws IOException, ServletException {

        try {

            String lang = LocaleHelper.toXsLanguage(i18n.getLocale());

            resp.setHeader("X-Frame-Options","SAMEORIGIN");
            resp.setHeader("Content-Language",lang);
            resp.setHeader("Cache-Control","no-cache");
            resp.setHeader("Pragma","no-cache");
            resp.setHeader("Expires","0");
            resp.setContentType("text/html; charset=utf-8");

            resp.getOutputStream().write("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" \"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n".getBytes("UTF-8"));

            XMLStreamWriter xsw = xmlOutputFactory.createXMLStreamWriter(resp.getOutputStream(),"UTF-8");

            xsw.setDefaultNamespace(XHTML_NS_URI);

            //xsw.writeStartDocument();

            xsw.writeStartElement("html");
            xsw.writeDefaultNamespace(XHTML_NS_URI);
            xsw.writeAttribute("lang",lang);
            xsw.writeAttribute("xml:lang",lang);
            xsw.writeStartElement("head");

            xsw.writeEmptyElement("meta");
            xsw.writeAttribute("http-equiv","Content-Type");
            xsw.writeAttribute("content","text/html; charset=utf-8");

            xsw.writeEmptyElement("link");
            xsw.writeAttribute("type","text/css");
            xsw.writeAttribute("rel","stylesheet");
            xsw.writeAttribute("href",css == null ? "oauth-login.css" : css);

            xsw.writeStartElement("script");
            xsw.writeCharacters("\nsetTimeout(window.close,3000);\n");
            xsw.writeEndElement();

            xsw.writeStartElement("body");

            if (scope != null) {

                xsw.writeStartElement("p");

                xsw.writeCharacters(i18n.formatString("authenticated-with-scope",scope));

                xsw.writeEndElement(); // </p>
            }

            if (error != null) {

                xsw.writeStartElement("p");
                xsw.writeCharacters(i18n.formatString("authentication-failed",domain));
                xsw.writeEndElement(); // </p>

                xsw.writeStartElement("p");
                xsw.writeCharacters(error.getError());

                if (error.getErrorDescription() != null) {
                    xsw.writeEmptyElement("br");
                    xsw.writeCharacters(error.getErrorDescription());
                }
                xsw.writeEndElement(); // </p>
            }

            xsw.writeEndElement(); // </body>

            xsw.writeEndElement(); // </html>

            xsw.writeEndDocument();
            xsw.close();

            resp.flushBuffer();

        } catch (XMLStreamException e) {

            throw new ServletException("Error setting XML stream writer",e);
        }

    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException {

        // state may be set by login iframe reload or successful response to oauth authorization request,
        String state = req.getParameter("state");
        String css = req.getParameter("css");
        String pi = req.getPathInfo();

        URI requestURI;

        try {
            requestURI = RequestHelper.getOriginalRequestUri(req);
        } catch (ParseException | URISyntaxException e) {
           throw new ServletException("Unable to parse full request URI for request to ["+RequestHelper.getRequestUrl(req)+"]",e);
        }

        if (log.isDebugEnabled()) {
            log.debug("Received request to [{}]",requestURI);
        }

        if (pi == null) {

            if (state == null) {
                log.info("Received request to [{}] without state parameter.",requestURI);
                resp.sendError(HttpServletResponse.SC_BAD_REQUEST);
                return;
            }

            AuthState authState = this.authStateCache.getAuthState(state);

            if (authState == null) {
                log.info("Received request to [{}] with invalid or expired state parameter.",requestURI);
                resp.sendError(HttpServletResponse.SC_BAD_REQUEST);
                return;
            }

            Locale locale = authState.getLocale();
            Messages i18n = OAuthMessages.getMesssages(locale);

            // successful response to oauth authorization request.
            String code = req.getParameter("code");

            if (code != null) {

                String domain = authState.getDomain();

                DomainManager domainManager = this.configurationService.getDomainManager(domain);
                DomainConfig domainConfig = domainManager.getDomainConfig();

                URI tokenUri = null;
                try {
                    tokenUri = domainManager.getTokenUri();

                    String scope = domainConfig.getScope();

                    log.info("Requesting token from [{}] upon request to [{}] with valid authentication code.",
                            tokenUri,requestURI);

                    URI redirectUri = new URI(requestURI.getScheme(),requestURI.getUserInfo(),requestURI.getHost(),requestURI.getPort(),
                            this.oauthHttpLoginService.getRedirectUrl(),
                            null,
                            requestURI.getFragment());


                    OAuthTokenResponse tokenResponse = this.oauthHttpClient.requestToken(tokenUri,
                            redirectUri.toString(),
                            state,scope,domainConfig.getClientCredentials(),code,domainConfig.getOptions());

                    authState.setResponse(tokenResponse);

                    this.renderAuthenticationResult(i18n,resp,css,authState.getDomain(),tokenResponse.getScope(),null);

                } catch (OAuthTokenErrorResponse e) {

                    log.error("Token request to [{}] with authentication code [{}] failed.",
                            tokenUri,code);

                    authState.setError(e);
                    this.renderAuthenticationResult(i18n,resp,css,authState.getDomain(),null,e);

                } catch (URISyntaxException e) {

                    log.error("Cannot build redirect URI for request to ["+requestURI+"].",e);

                    OAuthTokenErrorResponse oauthError =
                            new OAuthTokenErrorResponse("invalid_redirect_uri","configured redirect URI is malformed",null,null,null,null,null);

                    authState.setError(oauthError);

                    this.renderAuthenticationResult(i18n,resp,css,authState.getDomain(),null,oauthError);

                } catch (IllegalStateException e) {

                    log.error("OpenID configuration of domain ["+domainConfig.getDomain()+"] not loaded while requesting token location",e);
                    OAuthTokenErrorResponse oauthError = new OAuthTokenErrorResponse("openid-configuration-not-loaded",i18n);
                    this.renderAuthenticationResult(i18n,resp,css,authState.getDomain(),null,oauthError);
                }
            }
            else {

                String error = req.getParameter("error");

                if (error != null) {

                    // error response to oauth authorization request.
                    String errorDescription = req.getParameter("error_description");

                    log.info("Received request to [{}] with authentication error.",requestURI);

                    OAuthTokenErrorResponse oauthError = new OAuthTokenErrorResponse(error,errorDescription,null,null,null,null,null);

                    authState.setError(oauthError);

                    this.renderAuthenticationResult(i18n,resp,css,authState.getDomain(),null,oauthError);
                }
                else {
                    log.info("Received request to [{}] with neither an authentication code nor an error.",requestURI);
                    resp.sendError(HttpServletResponse.SC_BAD_REQUEST);
                }
            }
        }
        else {
            resp.sendError(HttpServletResponse.SC_NOT_FOUND);
        }
    }

    @Override
    public String getServletInfo() {

        return OAuthAuthServlet.class.getSimpleName();
    }

}
