/***********************************************************
 * $Id$
 *
 * OAuth Login Services of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 31.05.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.login.oauth;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonToken;

/**
 * A parse, which parses configuration objects, which are simple maps
 * of strings, numbers, booleans or string arrays.
 */
public class JsonConfigParser {

    private static final Logger log = LoggerFactory.getLogger(JsonConfigParser.class);

    /**
     * Parse a simple JSON configuration object.
     *
     * @param is A JSON input stream
     * @return A map with string keys and {@link String}, {@link Boolean},
     *         {@link Double}, arrays of strings or string->string maps.
     * @throws IOException On parse errors.
     */
    public static final Map<String,Object> parseConfig(InputStream is) throws IOException {

        try (JsonReader reader = new JsonReader(new InputStreamReader(is,"UTF-8"))) {

            Map<String,Object> ret = new HashMap<String,Object>();

            reader.beginObject();

            while (reader.hasNext()) {

                String name = reader.nextName();

                JsonToken tokenType = reader.peek();

                switch (tokenType) {

                case NULL:
                    reader.nextNull();
                    ret.put(name,null);
                    break;
                case BOOLEAN:
                    ret.put(name,reader.nextBoolean());
                    break;
                case NUMBER:
                    ret.put(name,reader.nextDouble());
                    break;
                case STRING:
                    ret.put(name,reader.nextString());
                    break;
                case BEGIN_ARRAY:

                    List<String> arr = new ArrayList<String>();

                    reader.beginArray();

                    while (reader.hasNext()) {
                        arr.add(reader.nextString());
                    }

                    reader.endArray();

                    ret.put(name,arr);
                    break;

                case BEGIN_OBJECT:

                    Map<String,String> map = new HashMap<String,String>();

                    reader.beginObject();

                    while (reader.hasNext()) {
                        String key = reader.nextName();
                        String value = reader.nextString();
                        map.put(key,value);
                    }

                    reader.endObject();

                    ret.put(name,map);
                    break;

                default:
                    throw new IllegalArgumentException("Unsupported configuration member type ["+tokenType+"].");
                }
            }

            reader.endObject();

            return ret;
        }

    }

    /**
     * Parse a simple JSON string map object.
     *
     * @param is A JSON input stream
     * @return A map with string keys and string values.
     * @throws IOException On parse errors.
     */
    public static final Map<String,Object> parseStringMap(InputStream is) throws IOException {

        try (JsonReader reader = new JsonReader(new InputStreamReader(is,"UTF-8"))) {

            Map<String,Object> ret = new HashMap<String,Object>();

            reader.beginObject();

            while (reader.hasNext()) {

                String name = reader.nextName();

                JsonToken tokenType = reader.peek();

                if (tokenType == JsonToken.NULL) {

                    reader.nextNull();
                    ret.put(name,null);
                }
                else if (tokenType == JsonToken.BOOLEAN) {
                    ret.put(name,reader.nextBoolean());
                }
                else if (tokenType == JsonToken.NUMBER) {
                    ret.put(name,reader.nextDouble());
                }
                else if (tokenType == JsonToken.STRING) {
                    ret.put(name,reader.nextString());
                }
                else if (tokenType == JsonToken.BEGIN_ARRAY) {

                    List<String> arr = new ArrayList<String>();

                    reader.beginArray();

                    while (reader.hasNext()) {
                        arr.add(reader.nextString());
                    }

                    reader.endArray();

                    ret.put(name,arr);
                }
                else {
                    log.warn("Ignoring unknown property [{}] in string map response.",name);
                    reader.skipValue();
                }
            }

            reader.endObject();

            return ret;
        }

    }

    public static final Map<String,String> parseAppUsers(String appUsers) {

        if (appUsers == null) {
            return null;
        }

        Map<String,String> ret = new HashMap<String,String>();

        String[] parts = appUsers.trim().split("\\s*,\\s*");

        for (String part:parts) {

            if (part.isEmpty()) {
                continue;
            }

            String[] appUser = part.split("\\s*:\\s*",2);

            if (appUser.length != 2) {
                throw new IllegalArgumentException("Invalid app user stanza ["+appUser+"] without colon.");
            }

            ret.put(appUser[0],appUser[1]);
        }

        return ret;
    }
}
